package org.jboss.cache.demo;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.Cache;
import org.jboss.cache.Fqn;
import org.jboss.cache.FqnComparator;
import org.jboss.cache.Node;

import javax.swing.*;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author Manik Surtani (<a href="mailto:manik@jboss.org">manik@jboss.org</a>)
 */
public class DataTreeRefresher
{
   private JTree tree;
   private Map<Fqn<String>, FqnTreeNode> treeNodes = new ConcurrentHashMap<Fqn<String>, FqnTreeNode>();
   private Cache<String, String> cache;
   private Log log = LogFactory.getLog(DataTreeRefresher.class);
   private JBossCacheDemo.NodeDataTableModel tableModel;

   public DataTreeRefresher(JTree tree, Cache<String, String> cache, JBossCacheDemo.NodeDataTableModel tableModel)
   {
      this.tree = tree;
      this.cache = cache;
      this.tableModel = tableModel;
      populateDataTree();
   }

   private void populateDataTree()
   {
      // initialise the root.
      FqnTreeNode node = new FqnTreeNode(Fqn.ROOT);
      // put this in the treeNodes map
      treeNodes.put(Fqn.ROOT, node);

      TreeModel model = new OrderedTreeModel(node);
      tree.setModel(model);

      // now traverse the cache nodes and populate the tree.
      populateChildren(cache.getRoot());

      repaint();
   }

   private void populateChildren(Node<String, String> cacheNode)
   {
      for (Node<String, String> cacheChild : cacheNode.getChildren())
      {
         getOrCreateTreeNode(cacheChild.getFqn());
         // now go deep
         populateChildren(cacheChild);
      }
   }

   private FqnTreeNode getOrCreateTreeNode(Fqn<String> f)
   {
      FqnTreeNode treeNode = treeNodes.get(f);
      if (treeNode == null)
      {
         treeNode = new FqnTreeNode(f);
         FqnTreeNode parent = getOrCreateTreeNode(f.getParent());
         parent.add(treeNode);
         treeNode.setParent(parent);
         treeNodes.put(f, treeNode);
      }
      return treeNode;
   }

   public void addNode(Fqn<String> f)
   {
      // node added!!
      log.debug("Adding node " + f, new Throwable());
      getOrCreateTreeNode(f);
   }

   public void removeNode(Fqn<String> f)
   {
      if (treeNodes.containsKey(f))
      {
         removeTreeNode(treeNodes.get(f));
      }
   }

   private void removeTreeNode(FqnTreeNode treeNode)
   {
      // clear children
      for (Enumeration en = treeNode.children(); en.hasMoreElements();)
      {
         FqnTreeNode child = (FqnTreeNode) en.nextElement();
         removeTreeNode(child);
      }

      if (!treeNode.getFqn().isRoot())
      {
         // then remove node
         treeNodes.remove(treeNode.getFqn());
         treeNode.removeFromParent();
      }
   }

   public void repaint()
   {
      TreePath path = tree.getSelectionPath();
      tableModel.setCurrentFqn(null);
      Map<String, String> m = Collections.emptyMap();
      tableModel.setData(m);
      ((DefaultTreeModel) tree.getModel()).reload();
      tree.setSelectionPath(path);
      if (path != null) while ((path = path.getParentPath()) != null) tree.expandPath(path);
   }

   class FqnTreeNode extends DefaultMutableTreeNode implements Comparable
   {
      private Fqn fqn;

      FqnTreeNode(Fqn fqn)
      {
         super(fqn.isRoot() ? "/" : fqn.getLastElement());
         this.fqn = fqn;
      }

      public Fqn getFqn()
      {
         return fqn;
      }

      public void setFqn(Fqn fqn)
      {
         this.fqn = fqn;
      }

      public int compareTo(Object o)
      {
         FqnTreeNode other = (FqnTreeNode) o;
         return FqnComparator.INSTANCE.compare(fqn, other.getFqn());
      }
   }

   class OrderedTreeModel extends DefaultTreeModel
   {
      public OrderedTreeModel(TreeNode root)
      {
         super(root);
      }

      public OrderedTreeModel(TreeNode root, boolean asksAllowsChildren)
      {
         super(root, asksAllowsChildren);
      }

      @Override
      public int getIndexOfChild(Object parent, Object child)
      {
         this.orderChildren(parent);
         return super.getIndexOfChild(parent, child);
      }

      @Override
      public Object getChild(Object parent, int index)
      {
         this.orderChildren(parent);
         return super.getChild(parent, index);
      }

      /**
       * Orders the children of a DefaultMutableTreeNode
       *
       * @param parent
       */
      private void orderChildren(Object parent)
      {
         if (parent == null) return;

         if (parent instanceof FqnTreeNode)
         {
            FqnTreeNode node = (FqnTreeNode) parent;
            ArrayList children = Collections.list(node.children());
            Collections.sort(children);
            node.removeAllChildren();
            Iterator childrenIterator = children.iterator();
            while (childrenIterator.hasNext()) node.add((FqnTreeNode) childrenIterator.next());
         }
      }
   }
}
