/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.cache.pojo.notification;

import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;

import javax.transaction.Transaction;

import org.jboss.cache.pojo.notification.annotation.Attached;
import org.jboss.cache.pojo.notification.annotation.Detached;
import org.jboss.cache.pojo.notification.annotation.FieldModified;
import org.jboss.cache.pojo.notification.annotation.ListModified;
import org.jboss.cache.pojo.notification.annotation.MapModified;
import org.jboss.cache.pojo.notification.annotation.PojoCacheListener;
import org.jboss.cache.pojo.notification.annotation.SetModified;
import org.jboss.cache.pojo.notification.annotation.TransactionCompleted;
import org.jboss.cache.pojo.notification.event.Event;
import org.jboss.cache.pojo.notification.event.TransactionCompletedEvent;

// $Id: TxGauranteedListener.java 4104 2007-07-03 01:46:32Z jgreene $

/**
 * A recoding Listener for notification test package.
 *
 * @author Jason T. Greene
 */
@PojoCacheListener
public class TxGauranteedListener
{
   private class TxEventQueue
   {
      private ConcurrentMap<Transaction, Queue<Event>> map = new ConcurrentHashMap<Transaction, Queue<Event>>();

      public void offer(Event event)
      {
         Queue<Event> queue = getQueue(event.getContext().getTransaction());
         queue.offer(event);
      }

      private Queue<Event> getQueue(Transaction transaction)
      {
         Queue<Event> queue = map.get(transaction);
         if (queue == null)
         {
            queue = new ConcurrentLinkedQueue<Event>();
            map.putIfAbsent(transaction, queue);
         }

         return queue;
      }

      public Queue<Event> takeAll(Transaction transaction)
      {
         return map.remove(transaction);
      }
   }

   private TxEventQueue events = new TxEventQueue();
   private Queue<Event> committed = new ConcurrentLinkedQueue<Event>();

   @SuppressWarnings("unchecked")
   public <T extends Event> T take(Class<T> t)
   {
      Event notification = committed.remove();
      if (!t.isInstance(notification))
         throw new IllegalStateException("Expected notification type: " + t.getSimpleName() + " but was: " + notification.getClass().getSimpleName());

      return (T) notification;
   }

   @Attached
   @Detached
   @FieldModified
   @ListModified
   @SetModified
   @MapModified
   public void handle(Event event)
   {
      events.offer(event);
   }

   @TransactionCompleted
   public void handleTx(TransactionCompletedEvent event)
   {
      Queue<Event> completed = events.takeAll(event.getContext().getTransaction());
      if (completed != null && event.isSuccessful())
         committed.addAll(completed);
   }
}