/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.cache.pojo.jmx;

import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertSame;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.List;
import java.util.Properties;

import javax.management.MBeanServerInvocationHandler;
import javax.transaction.TransactionManager;

import org.jboss.cache.Version;
import org.jboss.cache.config.BuddyReplicationConfig;
import org.jboss.cache.config.CacheLoaderConfig;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.EvictionConfig;
import org.jboss.cache.config.EvictionRegionConfig;
import org.jboss.cache.config.RuntimeConfig;
import org.jboss.cache.config.BuddyReplicationConfig.BuddyLocatorConfig;
import org.jboss.cache.config.CacheLoaderConfig.IndividualCacheLoaderConfig;
import org.jboss.cache.config.Configuration.CacheMode;
import org.jboss.cache.config.Configuration.NodeLockingScheme;
import org.jboss.cache.eviction.FIFOConfiguration;
import org.jboss.cache.eviction.FIFOPolicy;
import org.jboss.cache.eviction.LRUConfiguration;
import org.jboss.cache.eviction.LRUPolicy;
import org.jboss.cache.eviction.MRUConfiguration;
import org.jboss.cache.eviction.MRUPolicy;
import org.jboss.cache.loader.FileCacheLoader;
import org.jboss.cache.loader.SingletonStoreCacheLoader;
import org.jboss.cache.loader.jdbm.JdbmCacheLoader;
import org.jboss.cache.lock.IsolationLevel;
import org.jboss.cache.transaction.BatchModeTransactionManagerLookup;
import org.jboss.cache.xml.XmlHelper;
import org.jgroups.ChannelFactory;
import org.jgroups.JChannelFactory;
import org.testng.annotations.Test;
import org.w3c.dom.Element;

/**
 * Test of the CacheLegacyJmxWrapper.
 *
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 4456 $
 */
@Test(groups = {"functional"})
public class LegacyConfigurationTest extends PojoCacheJmxWrapperTestBase
{
   @SuppressWarnings("deprecation")
   public void testLocalCache() throws Exception
   {
      PojoCacheJmxWrapperMBean wrapper = new PojoCacheJmxWrapper();
      registerWrapper(wrapper);

      wrapper = (PojoCacheJmxWrapperMBean) MBeanServerInvocationHandler.newProxyInstance(mBeanServer, mBeanName, PojoCacheJmxWrapperMBean.class, false);

      wrapper.setBuddyReplicationConfig(getBuddyReplicationConfig());
      wrapper.setCacheLoaderConfig(getCacheLoaderConfig());
      wrapper.setCacheMode("REPL_SYNC");
      wrapper.setClusterName("LocalTest");
      wrapper.setClusterConfig(getClusterConfig());
      wrapper.setEvictionPolicyConfig(getEvictionPolicyConfig());
      wrapper.setFetchInMemoryState(false);
      wrapper.setInitialStateRetrievalTimeout(100);
      wrapper.setInactiveOnStartup(true);
      wrapper.setNodeLockingScheme("OPTIMISTIC");
      wrapper.setIsolationLevel("READ_UNCOMMITTED");
      wrapper.setLockAcquisitionTimeout(200);
      wrapper.setReplicationVersion("1.0.1");
      wrapper.setReplQueueInterval(15);
      wrapper.setReplQueueMaxElements(50);
      wrapper.setSyncReplTimeout(300);
      wrapper.setSyncCommitPhase(true);
      wrapper.setSyncRollbackPhase(true);
      wrapper.setTransactionManagerLookupClass(BatchModeTransactionManagerLookup.class.getName());
      wrapper.setExposeManagementStatistics(false);
      wrapper.setUseRegionBasedMarshalling(true);
      wrapper.setUseReplQueue(true);

      Configuration c = wrapper.getConfiguration();

      assertEquals("CacheMode", "REPL_SYNC", wrapper.getCacheMode());
      assertEquals("CacheMode", CacheMode.REPL_SYNC, c.getCacheMode());
      assertEquals("ClusterName", "LocalTest", wrapper.getClusterName());
      assertEquals("ClusterName", "LocalTest", c.getClusterName());
      assertEquals("FetchInMemoryState", false, wrapper.getFetchInMemoryState());
      assertEquals("FetchInMemoryState", false, c.isFetchInMemoryState());
      assertEquals("InitialStateRetrievalTimeout", 100, wrapper.getInitialStateRetrievalTimeout());
      assertEquals("InitialStateRetrievalTimeout", 100, c.getStateRetrievalTimeout());
      assertEquals("InactiveOnStartup", true, wrapper.isInactiveOnStartup());
      assertEquals("InactiveOnStartup", true, c.isInactiveOnStartup());
      assertEquals("NodeLockingScheme", "OPTIMISTIC", wrapper.getNodeLockingScheme());
      assertEquals("NodeLockingScheme", NodeLockingScheme.OPTIMISTIC, c.getNodeLockingScheme());
      assertEquals("IsolationLevel", "READ_UNCOMMITTED", wrapper.getIsolationLevel());
      assertEquals("IsolationLevel", IsolationLevel.READ_UNCOMMITTED, c.getIsolationLevel());
      assertEquals("LockAcquisitionTimeout", 200, wrapper.getLockAcquisitionTimeout());
      assertEquals("LockAcquisitionTimeout", 200, c.getLockAcquisitionTimeout());
      assertEquals("ReplicationVersion", "1.0.1", wrapper.getReplicationVersion());
      assertEquals("ReplicationVersion", Version.getVersionShort("1.0.1"), c.getReplicationVersion());
      assertEquals("ReplQueueInterval", 15, wrapper.getReplQueueInterval());
      assertEquals("ReplQueueInterval", 15, c.getReplQueueInterval());
      assertEquals("ReplQueueMaxElements", 50, wrapper.getReplQueueMaxElements());
      assertEquals("ReplQueueMaxElements", 50, c.getReplQueueMaxElements());
      assertEquals("SyncReplTimeout", 300, wrapper.getSyncReplTimeout());
      assertEquals("SyncReplTimeout", 300, c.getSyncReplTimeout());
      assertEquals("SyncCommitPhase", true, wrapper.getSyncCommitPhase());
      assertEquals("SyncCommitPhase", true, c.isSyncCommitPhase());
      assertEquals("SyncRollbackPhase", true, wrapper.getSyncRollbackPhase());
      assertEquals("SyncRollbackPhase", true, c.isSyncRollbackPhase());
      assertEquals("TransactionManagerLookupClass", BatchModeTransactionManagerLookup.class.getName(), wrapper.getTransactionManagerLookupClass());
      assertEquals("TransactionManagerLookupClass", BatchModeTransactionManagerLookup.class.getName(), c.getTransactionManagerLookupClass());
      assertEquals("ExposeManagementStatistics", false, wrapper.getExposeManagementStatistics());
      assertEquals("ExposeManagementStatistics", false, c.getExposeManagementStatistics());
      assertEquals("UseRegionBasedMarshalling", true, wrapper.getUseRegionBasedMarshalling());
      assertEquals("UseRegionBasedMarshalling", true, c.isUseRegionBasedMarshalling());
      assertEquals("UseReplQueue", true, wrapper.getUseReplQueue());
      assertEquals("UseReplQueue", true, c.isUseReplQueue());

      assertEquals("ClusterConfig", getClusterConfig().toString(), wrapper.getClusterConfig().toString());

      assertEquals("BuddyReplicationConfig", getBuddyReplicationConfig().toString(), wrapper.getBuddyReplicationConfig().toString());
      BuddyReplicationConfig brc = c.getBuddyReplicationConfig();
      assertEquals("BR enabled", true, brc.isEnabled());
      assertEquals("BR auto grav", false, brc.isAutoDataGravitation());
      assertEquals("BR remove find", false, brc.isDataGravitationRemoveOnFind());
      assertEquals("BR search backup", false, brc.isDataGravitationSearchBackupTrees());
      assertEquals("BR comm timeout", 600000, brc.getBuddyCommunicationTimeout());
      assertEquals("BR poolname", "testpool", brc.getBuddyPoolName());
      BuddyLocatorConfig blc = brc.getBuddyLocatorConfig();
      assertEquals("BR locator", "org.jboss.cache.buddyreplication.TestBuddyLocator", blc.getBuddyLocatorClass());
      Properties props = blc.getBuddyLocatorProperties();
      assertEquals("BR props", "2", props.get("numBuddies"));

      assertEquals("CacheLoaderConfig", getCacheLoaderConfig().toString(), wrapper.getCacheLoaderConfig().toString());
      CacheLoaderConfig clc = c.getCacheLoaderConfig();
      assertEquals("CL passivation", false, clc.isPassivation());
      assertEquals("CL passivation", true, clc.isShared());
      assertEquals("CL preload", "/foo", clc.getPreload());
      List<IndividualCacheLoaderConfig> iclcs = clc.getIndividualCacheLoaderConfigs();
      IndividualCacheLoaderConfig iclc = iclcs.get(0);
      assertEquals("CL0 class", FileCacheLoader.class.getName(), iclc.getClassName());
      assertEquals("CL0 async", false, iclc.isAsync());
      assertEquals("CL0 fetch", true, iclc.isFetchPersistentState());
      assertEquals("CL0 ignore", true, iclc.isIgnoreModifications());
      assertEquals("CL0 purge", true, iclc.isPurgeOnStartup());
      assertEquals("CL0 singleton", true, iclc.getSingletonStoreConfig().isSingletonStoreEnabled());
      assertEquals("CL0 singleton class", SingletonStoreCacheLoader.class.getName(), iclc.getSingletonStoreConfig().getSingletonStoreClass());
      iclc = iclcs.get(1);
      assertEquals("CL1 class", JdbmCacheLoader.class.getName(), iclc.getClassName());
      assertEquals("CL1 async", true, iclc.isAsync());
      assertEquals("CL1 fetch", false, iclc.isFetchPersistentState());
      assertEquals("CL1 ignore", false, iclc.isIgnoreModifications());
      assertEquals("CL1 purge", false, iclc.isPurgeOnStartup());
      assertEquals("CL1 singleton", false, iclc.getSingletonStoreConfig().isSingletonStoreEnabled());
      assertEquals("CL1 singleton class", SingletonStoreCacheLoader.class.getName(), iclc.getSingletonStoreConfig().getSingletonStoreClass());

      assertEquals("EvictionPolicyConfig", getEvictionPolicyConfig().toString(), wrapper.getEvictionPolicyConfig().toString());
      EvictionConfig ec = c.getEvictionConfig();
      assertEquals("EC queue size", 20000, ec.getDefaultEventQueueSize());
      assertEquals("EC wakeup", 5, ec.getWakeupIntervalSeconds());
      assertEquals("EC default pol", LRUPolicy.class.getName(), ec.getDefaultEvictionPolicyClass());
      List<EvictionRegionConfig> ercs = ec.getEvictionRegionConfigs();
      EvictionRegionConfig erc = ercs.get(0);
      assertEquals("ERC0 name", "/_default_", erc.getRegionName());
      assertEquals("ERC0 queue size", 1000, erc.getEventQueueSize());
      LRUConfiguration lru = (LRUConfiguration) erc.getEvictionPolicyConfig();
      assertEquals("EPC0 pol", LRUPolicy.class.getName(), lru.getEvictionPolicyClass());
      assertEquals("EPC0 maxnodes", 5000, lru.getMaxNodes());
      assertEquals("EPC0 ttl", 1000, lru.getTimeToLiveSeconds());
      erc = ercs.get(1);
      assertEquals("ERC1 name", "/org/jboss/data", erc.getRegionName());
      assertEquals("ERC1 queue size", 20000, erc.getEventQueueSize());
      FIFOConfiguration fifo = (FIFOConfiguration) erc.getEvictionPolicyConfig();
      assertEquals("EPC1 pol", FIFOPolicy.class.getName(), fifo.getEvictionPolicyClass());
      assertEquals("EPC1 maxnodes", 5000, fifo.getMaxNodes());
      erc = ercs.get(2);
      assertEquals("ERC2 name", "/test", erc.getRegionName());
      assertEquals("ERC2 queue size", 20000, erc.getEventQueueSize());
      MRUConfiguration mru = (MRUConfiguration) erc.getEvictionPolicyConfig();
      assertEquals("EPC2 pol", MRUPolicy.class.getName(), mru.getEvictionPolicyClass());
      assertEquals("EPC2 maxnodes", 10000, mru.getMaxNodes());
      erc = ercs.get(3);
      assertEquals("ERC3 name", "/maxAgeTest", erc.getRegionName());
      assertEquals("ERC3 queue size", 20000, erc.getEventQueueSize());
      lru = (LRUConfiguration) erc.getEvictionPolicyConfig();
      assertEquals("EPC3 pol", LRUPolicy.class.getName(), lru.getEvictionPolicyClass());
      assertEquals("EPC3 maxnodes", 10000, lru.getMaxNodes());
      assertEquals("EPC3 maxage", 10, lru.getMaxAgeSeconds());
      assertEquals("EPC3 ttl", 8, lru.getTimeToLiveSeconds());

   }

   public void testRuntimeConfig() throws Exception
   {
      PojoCacheJmxWrapperMBean wrapper = new PojoCacheJmxWrapper();
      registerWrapper(wrapper);

      wrapper = (PojoCacheJmxWrapperMBean) MBeanServerInvocationHandler.newProxyInstance(mBeanServer, mBeanName, PojoCacheJmxWrapperMBean.class, false);

      // Fake a TM by making a bogus proxy
      TransactionManager tm = (TransactionManager) Proxy.newProxyInstance(getClass().getClassLoader(),
                                                                          new Class[]{TransactionManager.class}, new MockInvocationHandler());
      wrapper.setTransactionManager(tm);
      ChannelFactory cf = new JChannelFactory();
      wrapper.setMuxChannelFactory(cf);

      RuntimeConfig rc = wrapper.getConfiguration().getRuntimeConfig();

      assertSame("Same TM", tm, wrapper.getTransactionManager());
      assertSame("Same TM", tm, rc.getTransactionManager());
      assertSame("Same ChannelFactory", cf, wrapper.getMuxChannelFactory());
      assertSame("Same ChannelFactory", cf, rc.getMuxChannelFactory());
   }

   protected static Element getBuddyReplicationConfig() throws Exception
   {
      String xmlString = "<config><buddyReplicationEnabled>true</buddyReplicationEnabled>\n" +
                         "          <buddyCommunicationTimeout>600000</buddyCommunicationTimeout>\n" +
                         "          <buddyLocatorClass>org.jboss.cache.buddyreplication.TestBuddyLocator</buddyLocatorClass>\n" +
                         "          <buddyLocatorProperties>numBuddies = 2</buddyLocatorProperties>\n" +
                         "          <buddyPoolName>testpool</buddyPoolName>" +
                         "          <autoDataGravitation>false</autoDataGravitation>\n" +
                         "          <dataGravitationRemoveOnFind>false</dataGravitationRemoveOnFind>\n" +
                         "          <dataGravitationSearchBackupTrees>false</dataGravitationSearchBackupTrees>" +
                         "</config>";
      return XmlHelper.stringToElement(xmlString);
   }

   protected static Element getCacheLoaderConfig() throws Exception
   {
      String xml = "<config>\n" +
                   "<passivation>false</passivation>\n" +
                   "<preload>/foo</preload>\n" +
                   "<shared>true</shared>\n" +
                   "<cacheloader>\n" +
                   "<class>org.jboss.cache.loader.FileCacheLoader</class>\n" +
                   "<properties>" +
                   " location=/tmp\n" +
                   "</properties>\n" +
                   "<async>false</async>\n" +
                   "<fetchPersistentState>true</fetchPersistentState>\n" +
                   "<ignoreModifications>true</ignoreModifications>\n" +
                   "<purgeOnStartup>true</purgeOnStartup>\n" +
                   "<singletonStore>" +
                   "<enabled>true</enabled>" +
                   "</singletonStore>" +
                   "</cacheloader>\n" +
                   "<cacheloader>\n" +
                   "<class>org.jboss.cache.loader.jdbm.JdbmCacheLoader</class>\n" +
                   "<properties>" +
                   " location=/home/bstansberry\n" +
                   "</properties>\n" +
                   "<async>true</async>\n" +
                   "<fetchPersistentState>false</fetchPersistentState>\n" +
                   "<ignoreModifications>false</ignoreModifications>\n" +
                   "<purgeOnStartup>false</purgeOnStartup>\n" +
                   "<singletonStore>" +
                   "<enabled>false</enabled>" +
                   "</singletonStore>" +
                   "</cacheloader>\n" +
                   "</config>";
      return XmlHelper.stringToElement(xml);
   }

   protected static Element getEvictionPolicyConfig() throws Exception
   {
      String xml = "<config>\n" +
                   "<attribute name=\"wakeUpIntervalSeconds\">5</attribute>\n" +
                   "<attribute name=\"eventQueueSize\">20000</attribute>\n" +
                   "<attribute name=\"policyClass\">org.jboss.cache.eviction.LRUPolicy</attribute>\n" +
                   "<region name=\"/_default_\" eventQueueSize=\"1000\">\n" +
                   "   <attribute name=\"maxNodes\">5000</attribute>\n" +
                   "   <attribute name=\"timeToLiveSeconds\">1000</attribute>\n" +
                   "</region>\n" +
                   "<region name=\"/org/jboss/data\" policyClass=\"org.jboss.cache.eviction.FIFOPolicy\">\n" +
                   "   <attribute name=\"maxNodes\">5000</attribute>\n" +
                   "</region>\n" +
                   "<region name=\"/test/\" policyClass=\"org.jboss.cache.eviction.MRUPolicy\">\n" +
                   "   <attribute name=\"maxNodes\">10000</attribute>\n" +
                   "</region>\n" +
                   "<region name=\"/maxAgeTest/\">\n" +
                   "   <attribute name=\"maxNodes\">10000</attribute>\n" +
                   "   <attribute name=\"timeToLiveSeconds\">8</attribute>\n" +
                   "   <attribute name=\"maxAgeSeconds\">10</attribute>\n" +
                   "</region>\n" +
                   " </config>\n";
      return XmlHelper.stringToElement(xml);
   }

   protected static Element getClusterConfig() throws Exception
   {
      String xml =
            "<config>\n" +
            "  <UDP mcast_addr=\"228.10.10.10\"\n" +
            "           mcast_port=\"45588\"\n" +
            "           tos=\"8\"\n" +
            "           ucast_recv_buf_size=\"20000000\"\n" +
            "           ucast_send_buf_size=\"640000\"\n" +
            "           mcast_recv_buf_size=\"25000000\"\n" +
            "           mcast_send_buf_size=\"640000\"\n" +
            "           loopback=\"false\"\n" +
            "           discard_incompatible_packets=\"true\"\n" +
            "           max_bundle_size=\"64000\"\n" +
            "           max_bundle_timeout=\"30\"\n" +
            "           use_incoming_packet_handler=\"true\"\n" +
            "           ip_ttl=\"2\"\n" +
            "           enable_bundling=\"false\"\n" +
            "           enable_diagnostics=\"true\"\n" +
            "           use_concurrent_stack=\"true\"\n" +
            "           thread_naming_pattern=\"pl\"\n" +
            "           thread_pool.enabled=\"true\"\n" +
            "           thread_pool.min_threads=\"1\"\n" +
            "           thread_pool.max_threads=\"25\"\n" +
            "           thread_pool.keep_alive_time=\"30000\"\n" +
            "           thread_pool.queue_enabled=\"true\"\n" +
            "           thread_pool.queue_max_size=\"10\"\n" +
            "           thread_pool.rejection_policy=\"Run\"\n" +
            "           oob_thread_pool.enabled=\"true\"\n" +
            "           oob_thread_pool.min_threads=\"1\"\n" +
            "           oob_thread_pool.max_threads=\"4\"\n" +
            "           oob_thread_pool.keep_alive_time=\"10000\"\n" +
            "           oob_thread_pool.queue_enabled=\"true\"\n" +
            "           oob_thread_pool.queue_max_size=\"10\"\n" +
            "           oob_thread_pool.rejection_policy=\"Run\"/>\n" +
            "      <PING timeout=\"2000\" num_initial_members=\"3\"/>\n" +
            "      <MERGE2 max_interval=\"30000\" min_interval=\"10000\"/>\n" +
            "      <FD_SOCK/>\n" +
            "      <FD timeout=\"10000\" max_tries=\"5\" shun=\"true\"/>\n" +
            "      <VERIFY_SUSPECT timeout=\"1500\"/>\n" +
            "      <pbcast.NAKACK max_xmit_size=\"60000\"\n" +
            "                     use_mcast_xmit=\"false\" gc_lag=\"0\"\n" +
            "                     retransmit_timeout=\"300,600,1200,2400,4800\"\n" +
            "                     discard_delivered_msgs=\"true\"/>\n" +
            "      <UNICAST timeout=\"300,600,1200,2400,3600\"/>\n" +
            "      <pbcast.STABLE stability_delay=\"1000\" desired_avg_gossip=\"50000\"\n" +
            "                     max_bytes=\"400000\"/>\n" +
            "      <pbcast.GMS print_local_addr=\"true\" join_timeout=\"5000\"\n" +
            "                  join_retry_timeout=\"2000\" shun=\"false\"\n" +
            "                  view_bundling=\"true\" view_ack_collection_timeout=\"5000\"/>\n" +
            "      <FRAG2 frag_size=\"60000\"/>\n" +
            "      <pbcast.STREAMING_STATE_TRANSFER use_reading_thread=\"true\"/>\n" +
            "      <pbcast.FLUSH timeout=\"0\"/>\n" +
            "</config>";
      return XmlHelper.stringToElement(xml);
   }

   class MockInvocationHandler implements InvocationHandler
   {

      public Object invoke(Object proxy, Method method, Object[] args) throws Throwable
      {
         return null;
      }

   }
}
