package org.jboss.cache.search;

import org.apache.lucene.search.Sort;
import org.apache.lucene.search.Filter;
import org.hibernate.search.FullTextFilter;

import java.util.List;

/**
 * A cache-query is what will be returned when the createQuery() method is run. This object can have methods such
 * as list, setFirstResult,setMaxResults, setFetchSize, getResultSize and setSort.
 *
 * @author Manik Surtani (<a href="mailto:manik@jboss.org">manik@jboss.org</a>)
 * @see org.jboss.cache.search.SearchableCache#createQuery(org.apache.lucene.search.Query) 
 */
public interface CacheQuery extends Iterable
{
   /**
    * Returns the results of a search as a list.
    *
    * @return list of objects that were found from the search.
    */

   List<Object> list();

   /**
    * Returns the results of a search as a {@link QueryResultIterator}.
    *
    * @return a QueryResultIterator which can be used to iterate through the results that were found.
    */

   QueryResultIterator iterator();

   /**
    * Sets a result with a given index to the first result.
    *
    * @param index of result to be set to the first.
    * @throws IllegalArgumentException if the index given is less than zero.
    */

   void setFirstResult(int index);

   /**
    * Sets the maximum number of results to the number passed in as a parameter.
    *
    * @param numResults that are to be set to the maxResults.
    */

   void setMaxResults(int numResults);

   /**
    * Defines scrollable result fetch size
    *
    * @param size to be set
    */

   void setFetchSize(int size);

   /**
    * Gets the integer number of results.
    *
    * @return integer number of results.
    */

   int getResultSize();

   /**
    * Allows lucene to sort the results. Integers are sorted in descending order.
    *
    * @param s - lucene sort object
    */

   void setSort(Sort s);

   /**
    * Enable a given filter by its name.
    *
    * @param name of filter.
    * @return a FullTextFilter object.
    */
   public FullTextFilter enableFullTextFilter(String name);


   /**
    * Disable a given filter by its name.
    *
    * @param name of filter.
    */
   public void disableFullTextFilter(String name);

   /**
    * Allows lucene to filter the results.
    *
    * @param f - lucene filter
    */

   public void setFilter(Filter f);



}
