package org.jboss.cache.search;

import org.hibernate.search.impl.SearchFactoryImpl;
import org.hibernate.search.backend.TransactionContext;
import org.hibernate.search.backend.Work;
import org.hibernate.search.backend.WorkType;
import org.hibernate.search.engine.SearchFactoryImplementor;
import org.jboss.cache.notifications.annotation.CacheListener;
import org.jboss.cache.notifications.annotation.NodeModified;
import org.jboss.cache.notifications.event.NodeModifiedEvent;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Map;

/**
 * @author Navin Surtani  - navin@surtani.org
 *
 * Listener class for changes made to the cache. This listener makes changes if it is a org.jboss.cache being used. 
 */
@CacheListener
public class SearchableCoreListener
{
   private SearchFactoryImplementor searchFactory;
   private static final Log log = LogFactory.getLog(SearchableCoreListener.class);

   public SearchableCoreListener(SearchFactoryImplementor searchFactory)
   {
      
      this.searchFactory = searchFactory;
   }

   /**
    * Takes in a NodeModifiedEvent and updates the Lucene indexes using methods on the NodeModifiedEvent class.
    *
    * @param event that has occured - or a node that has been changed. {@link org.jboss.cache.notifications.event.NodeModifiedEvent}
    * @throws InvalidKeyException if an invalid key is passed in.
    */
   
   @NodeModified
   public void updateLuceneIndexes(NodeModifiedEvent event) throws InvalidKeyException
   {

      if (log.isTraceEnabled()) log.trace("You have entered the SearchableListener");
      if (!event.isPre())
      {
         if (log.isTraceEnabled()) log.trace("event.isPre is false. Going to start updating indexes");
         switch (event.getModificationType())
         {
            case PUT_MAP:
            case PUT_DATA:
               if (log.isTraceEnabled()) log.trace("put() has been called on cache. Going to handle the data.");
               handlePutData(event, searchFactory);
               break;
            case REMOVE_DATA:
               handleDeleteData(event, searchFactory);
               break;
         }
      }
   }

      /**
    * If the modification type is PUT_MAP or PUT_DATA then this method will be called.
    * Takes in the event as a parameter
    *
    * @param event that has occured - or a node that has been changed. {@link org.jboss.cache.notifications.event.NodeModifiedEvent}
    * @param searchFactory - a SearchFactoryImpl instance.
    * @throws InvalidKeyException
    */


   protected void handlePutData(NodeModifiedEvent event, SearchFactoryImplementor searchFactory) throws InvalidKeyException
   {
      Map dataMap = event.getData();
      if (log.isTraceEnabled()) log.trace("Called event.getData() and saved to a Map");


      TransactionContext ctx = new NodeModifiedTransactionContext(event);

      for (Object key : dataMap.keySet())
      {
         CacheEntityId cacheEntityId = new CacheEntityId(event.getFqn(), (String) key);
         if (log.isTraceEnabled()) log.trace("Created new CacheEntityId");

         String fqnString = cacheEntityId.getFqn().toString();  // Vars for logging
         String keyString = (String) key;


         searchFactory.getWorker().performWork(new Work(dataMap.get(key), cacheEntityId.getDocumentId(), WorkType.DELETE), ctx);
         searchFactory.getWorker().performWork(new Work(dataMap.get(key), cacheEntityId.getDocumentId(), WorkType.ADD), ctx);


         if (log.isTraceEnabled())
         {
            log.debug("Added your object into Lucene with Fqn " + fqnString + " and key " + keyString);
         }


      }

   }



   /**
    * If the modification type is DELETE_DATA then this method will be called.
    * Takes in the event as a parameter
    *
    * @param event that has occured - or a node that has been changed. {@link org.jboss.cache.notifications.event.NodeModifiedEvent}
    * @param searchFactory - a SearchFactoryImpl instance.
    * @throws InvalidKeyException
    */
   protected void handleDeleteData(NodeModifiedEvent event, SearchFactoryImplementor searchFactory) throws InvalidKeyException
   {
      Map dataMap = event.getData();

      TransactionContext ctx = new NodeModifiedTransactionContext(event);

      for (Object key : dataMap.keySet())
      {
         CacheEntityId cacheEntityId = new CacheEntityId(event.getFqn(), (String) key);

         String fqnString = cacheEntityId.getFqn().toString();
         String keyString = (String) key;


         searchFactory.getWorker().performWork(new Work(dataMap.get(key), cacheEntityId.getDocumentId(), WorkType.DELETE), ctx);
         if (log.isTraceEnabled()) log.trace("Deleted your object from Lucene with Fqn " + fqnString + " and key " + keyString);


      }

   }

}
