/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.framework.server;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;
import java.util.Arrays;

import org.jboss.ha.framework.interfaces.CachableMarshalledValue;
import org.jboss.ha.framework.interfaces.ObjectStreamSource;

/**
 * Variation on the standard JBoss <code>org.jboss.invocation.MarshalledValue</code>
 * that only converts the wrapped object to a serialized form during
 * serialization itself. Allows the marshalled value to be cached without adding
 * the memory overhead of maintaining a byte[] version of the wrapped object.
 *
 * @author Brian Stansberry
 * @version $Revision: 37459 $
 */
public class SimpleCachableMarshalledValue
   implements java.io.Externalizable, CachableMarshalledValue
{
   /** 
    * Value returned by {@link #hashCode()} if <code>true</code> was passed
    * as the <code>ignoreHashCode</code> param to the constructor.
    */
   public static final int OUR_HASH_CODE = Integer.MAX_VALUE;
   
   /** 
    * Value returned by {@link #hashCode()} if <code>false</code> was passed
    * to the constructor as the <code>ignoreHashCode</code> param and <code>null</code>
    * was passed as the <code>obj</code> param.
    */
   public static final int NULL_HASH_CODE = 0;
   
   /**
    * The serialized form of the value.
    */
   private byte[] serializedForm;
   
   /**
    * The raw form of the passed in object.
    */
   private Serializable raw;
   
   private int hashCode = OUR_HASH_CODE;
   
   private transient ObjectStreamSource objectStreamSource;

   /**
    * Exposed for externalization.
    */
   public SimpleCachableMarshalledValue()
   {
      super();
   }

   /**
    * Create a new SimpleCachableMarshalledValue. Same as 
    * <code SimpleCachableMarshalledValue(obj, false)</code>.
    * 
    * @param obj the object to wrap
    */
   public SimpleCachableMarshalledValue(Serializable obj)
   {
      this(obj, false);
   }

   /**
    * Create a new SimpleCachableMarshalledValue.
    * 
    * @param obj the object to wrap. May be <code>null</code>
    * 
    * @param ignoreHashCode <code>true</code> if this object is not expected to be
    *               used as a key in any hashmap/HashSet like structures, where its hashcode
    *               should be the same as <code>obj</code>'s hashcode. If <code>false</code>
    *               this constructor will call <code>obj.hashCode()</code> and
    *               cache the value.
    */
   public SimpleCachableMarshalledValue(Serializable obj, boolean ignoreHashCode)
   {
      this.raw = obj;
      if (!ignoreHashCode)
      {
         if (this.raw != null)
         {
            this.hashCode = raw.hashCode();
         }
         else
         {
            this.hashCode = NULL_HASH_CODE;
         }
      }
   }

   /**
    * Create a new SimpleCachableMarshalledValue. Same as 
    * <code SimpleCachableMarshalledValue(obj, streamSource, false)</code>.
    * 
    * @param obj the object to wrap. May be <code>null</code>
    * @param streamSource source for object streams used for serialization and 
    *                     deserialization. May be <code>null</code>
    */
   public SimpleCachableMarshalledValue(Serializable obj, ObjectStreamSource streamSource)
   {
      this(obj, streamSource, false);
   }

   /**
    * Create a new SimpleCachableMarshalledValue.
    * 
    * @param obj the object to wrap. May be <code>null</code>
    * @param streamSource source for object streams used for serialization and 
    *                     deserialization. May be <code>null</code>
    * @param ignoreHashCode <code>true</code> if this object is not expected to be
    *               used as a key in any hashmap/HashSet like structures, where its hashcode
    *               should be the same as <code>obj</code>'s hashcode. If <code>false</code>
    *               this constructor will call <code>obj.hashCode()</code> and
    *               cache the value.
    */
   public SimpleCachableMarshalledValue(Serializable obj, ObjectStreamSource streamSource, boolean ignoreHashCode)
   {
      this(obj, ignoreHashCode);
      this.objectStreamSource = streamSource;
   }

   public synchronized Serializable get() throws IOException, ClassNotFoundException
   {
      if (raw == null)
      {
         if (serializedForm != null)
         {
            ByteArrayInputStream bais = new ByteArrayInputStream(serializedForm);
            ObjectInput mvis = getObjectStreamSource().getObjectInput(bais);
            try
            {
               raw =  (Serializable) mvis.readObject();
               serializedForm = null;
            }
            finally
            {
               mvis.close();
            }
         }         
      }
      return raw;
   }

   public synchronized byte[] toByteArray() throws IOException
   {
      if (serializedForm == null)
      {
         serializedForm = serialize();
      }
      
      raw = null;
      
      return serializedForm;
   }
   
   public Serializable peekUnderlyingObject()
   {
      return raw;
   }
   
   public byte[] peekSerializedForm()
   {
      return serializedForm;
   }

   public ObjectStreamSource getObjectStreamSource()
   {
      if (objectStreamSource == null)
      {
         objectStreamSource = new MarshalledValueObjectStreamSource();
      }
      return objectStreamSource;
   }

   public void setObjectStreamSource(ObjectStreamSource objectStreamSource)
   {
      this.objectStreamSource = objectStreamSource;
   }

   /**
    * Returns either the hashcode of the wrapped object or {@link Integer#MAX_VALUE}; 
    * which depends on whether the <code>ignoreHashCode</code>
    * param was set to <code>true</code> in this object's constructor.
    *
    * @return either the hashcode of the wrapped object or {@link Integer#MAX_VALUE}
    */
   @Override
   public int hashCode()
   {
      return hashCode;
   }

   @Override
   public boolean equals(Object obj)
   {
      if( this == obj )
         return true;

      boolean equals = false;
      if( obj instanceof SimpleCachableMarshalledValue )
      {
         SimpleCachableMarshalledValue mv = (SimpleCachableMarshalledValue) obj;
         if (raw != null && mv.raw != null)
         {
            equals = raw.equals(mv.raw);
         }
         else if (serializedForm != null && mv.serializedForm != null)
         {
            if( serializedForm == mv.serializedForm )
            {
               equals = true;
            }
            else
            {
               equals = Arrays.equals(serializedForm, mv.serializedForm);
            }
         }
         else
         {
            byte[] us = null;
            byte[] them = null;
            try
            {
               us = serialize();
               them = mv.serialize();
               equals = Arrays.equals(us, them);
            }
            catch (IOException e)
            {
               throw new RuntimeException("Cannot serialize " + ((us == null && raw != null) ? "us" : "them"), e);
            }
         }
      }
      return equals;
   }
   
   @Override
   public String toString()
   {
      StringBuilder sb = new StringBuilder(getClass().getName());
      sb.append("{raw=");
      if (raw == null)
      {
         sb.append("null");
      }
      else
      {
         sb.append(raw.getClass().getName());
         sb.append('@');
         sb.append(System.identityHashCode(raw));
      }
      sb.append("serialized=");
      sb.append(serializedForm==null ? "false" : "true");
      sb.append('}');
      return sb.toString();
   }
   
   /**
    * The object implements the readExternal method to restore its
    * contents by calling the methods of DataInput for primitive
    * types and readObject for objects, strings and arrays.  The
    * readExternal method must read the values in the same sequence
    * and with the same types as were written by writeExternal.
    *
    * @param in the stream to read data from in order to restore the object
    * 
    * @throws IOException              if I/O errors occur
    * @throws ClassNotFoundException   If the class for an object being
    *                                  restored cannot be found.
    */
   public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
   {
      int length = in.readInt();
      serializedForm = null;
      if( length > 0 )
      {
         serializedForm = new byte[length];
         in.readFully(serializedForm);
      }
      hashCode = in.readInt();
   }

   /**
    * The object implements the writeExternal method to save its contents
    * by calling the methods of DataOutput for its primitive values or
    * calling the writeObject method of ObjectOutput for objects, strings,
    * and arrays.
    *
    * @serialData Overriding methods should use this tag to describe
    *            the data layout of this Externalizable object.
    *            List the sequence of element types and, if possible,
    *            relate the element to a public/protected field and/or
    *            method of this Externalizable class.
    *
    * @param out    the stream to write the object to
    * 
    * @throws IOException   Includes any I/O exceptions that may occur
    */
   public synchronized void writeExternal(ObjectOutput out) throws IOException
   {
      byte[] bytes = serialize();
      int length = bytes != null ? bytes.length : 0;
      out.writeInt(length);
      if( length > 0 )
      {
         out.write(bytes);
      }
      out.writeInt(hashCode);
   }
   
   private byte[] serialize() throws IOException
   {
      byte[] result = serializedForm;
      if (result == null)
      {
         if (raw != null)
         {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ObjectOutput mvos = getObjectStreamSource().getObjectOutput(baos);
            mvos.writeObject(raw);
            mvos.flush();
            try
            {
               result = baos.toByteArray();
            }
            finally
            {
               mvos.close();
            }
         }
         
      }
      return result;
   }
}
