/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.ha.framework.interfaces;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

/** 
 *   HAPartition is an abstraction of the communication framework, providing access
 *   to clustered services such as Distributed State, Distributed Replicant Manager
 *   and RPC.
 *  
 *   @author  <a href="mailto:bill@burkecentral.com">Bill Burke</a>.
 *   @author  <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 *   @author  <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 *   @version $Revision: 104414 $
 *
 * <p><b>Revisions:</b><br>
 * <p><b>28.07.2002 - Sacha Labourey:</b>
 * <ul>
 * <li> Added network-partition merge callback for listeners</li>
 * </ul>
 */

public interface HAPartition extends GroupRpcDispatcher, GroupMembershipNotifier, GroupStateTransferService
{
   // *******************************
   // *******************************
   // Partition information accessors
   // *******************************
   // *******************************
   
   /**
    * Return the name of this node in the current partition. The name is
    * dynamically determined by the partition. The name will be the String
    * returned by <code>getClusterNode().getName()</code>.
    * 
    * @return The node name
    * 
    * @see #getClusterNode()
    */   
   public String getNodeName();
   /**
    * The name of the partition.
    *
    * @return Name of the current partition
    */   
   public String getPartitionName();

   /**
    * Accessor to the DistributedReplicantManager (DRM) that is linked to this partition.
    * @return the DistributedReplicantManager
    */   
   public DistributedReplicantManager getDistributedReplicantManager();
   
   /**
    * Accessor to the DistributedState (DS) that is linked to this partition.
    * @return the DistributedState service
    * 
    * @deprecated Use JBoss Cache for distributed caching
    */   
   public DistributedState getDistributedStateService ();

   // ***************************
   // ***************************
   // RPC multicast communication
   // ***************************
   // ***************************

   
   /**
    * Invoke an RPC call on all nodes of the partition/cluster and return their
    * response values as a list. This convenience method is equivalent to 
    * {@link #callMethodOnCluster(String, String, Object[], Class[], Class, boolean, ResponseFilter, long, boolean) 
    * callAsynchMethodOnCluster(serviceName, methodName, args, types, Object.class, excludeSelf, null, methodTimeout, false)} 
    * where <code>methodTimeout</code> is the value returned by {@link #getMethodCallTimeout()}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>false</code> if the RPC must also be made on the current
    * node of the partition, <code>true</code> if only on remote nodes
    *
    * @return an array of responses from nodes that invoked the RPC
    * 
    * @deprecated Use {@link GroupRpcDispatcher#callMethodOnCluster(String, String, Object[], Class[], boolean) the superinterface version}
    */
   @Deprecated
   @Override
   public ArrayList<?> callMethodOnCluster(String serviceName, String methodName,
         Object[] args, Class<?>[] types, boolean excludeSelf) throws InterruptedException;
   
   /**
    * Invoke a synchronous RPC call on all nodes of the partition/cluster and return their
    * response values as a list. This convenience method is equivalent to 
    * {@link #callMethodOnCluster(String, String, Object[], Class[], boolean, ResponseFilter, boolean) 
    * callAsynchMethodOnCluster(serviceName, methodName, args, types, Object.class, excludeSelf, filter, methodTimeout, false)} 
    * where <code>methodTimeout</code> is the value returned by {@link #getMethodCallTimeout()}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>false</code> if the RPC must also be made on the current
    * node of the partition, <code>true</code> if only on remote nodes
    * @param filter response filter instance which allows for early termination 
    * of the synchronous RPC call. Can be <code>null</code>.
    * 
    * @return an array of responses from remote nodes
    * 
    * @deprecated Use {@link GroupRpcDispatcher#callMethodOnCluster(String, String, Object[], Class[], boolean, ResponseFilter) the superinterface version}
    */
   @Deprecated
   @Override
   public ArrayList<?> callMethodOnCluster(String serviceName, String methodName,
         Object[] args, Class<?>[] types, boolean excludeSelf, ResponseFilter filter) throws InterruptedException;
   // *************************
   // *************************
   // State transfer management
   // *************************
   // *************************
   
   
   /**
    * State management is highly important for clustered services. Consequently, services that wish to manage their state
    * need to subscribe to state transfer events. When a service is started on a cluster node, state is pushed from another node to the  
    * new node.  When another node starts, the node may be asked to provide its state to initialise the newly started node.
    *
    * @deprecated Use the {@link GroupStateTransferService} API
    */   
   @Deprecated
   public interface HAPartitionStateTransfer extends StateTransferProvider
   {
      /**
       * This callback method is called when a new service starts on a new node; the state that it should hold is transfered to it through this callback.
       * @param newState The serialized representation of the state of the new service.
       */      
      public void setCurrentState(Serializable newState);
   }
   
   /**
    * Register a service that will participate in state transfer protocol and receive callbacks
    * @param serviceName Name of the service that subscribes for state transfer events. This name must be identical for all identical services in the cluster.
    * @param subscriber Object implementing {@link HAPartitionStateTransfer} and providing or receiving state transfer callbacks
    *
    * @deprecated Use the {@link GroupStateTransferService} API
    */   
   @Deprecated
   public void subscribeToStateTransferEvents (String serviceName, HAPartition.HAPartitionStateTransfer subscriber);
   
   /**
    * Unregister a service from state transfer callbacks.
    * @param serviceName Name of the service that participates in the state transfer protocol
    * @param subscriber Service implementing the state transfer callback methods
    *
    * @deprecated Use the {@link GroupStateTransferService} API
    */   
   @Deprecated
   public void unsubscribeFromStateTransferEvents (String serviceName, HAPartition.HAPartitionStateTransfer subscriber);

   // *************************
   // *************************
   // Group Membership listeners
   // *************************
   // *************************
   
   /**
    * Listener for notifications issued when a new node joins the cluster or an 
    * existing node leaves the cluster (or simply dies).
    * 
    * @deprecated Use {@link GroupMembershipListener}
    */
   @Deprecated
   public interface HAMembershipListener
   {
      /** Called when a partition topology change occurs. This callback may be
       * made using the thread that carries messages up from the network and so you 
       * should not execute new cluster calls using this thread. If you need to 
       * do that implement the 
       * {@link AsynchHAMembershipListener asynchronous version of this interface}.
       *
       * @param deadMembers A list of nodes that have died since the previous view
       * @param newMembers A list of nodes that have joined the partition since the previous view
       * @param allMembers A list of nodes that built the current view
       */      
      public void membershipChanged(Vector<ClusterNode> deadMembers, Vector<ClusterNode> newMembers, Vector<ClusterNode> allMembers);
   }

   /** 
    * A tagging interface for HAMembershipListener implementations that require
    * notifications to be issued by a thread separate from the thread that 
    * carries messages up from the network. The ordering of view changes is 
    * preserved, but listeners are free to execute cluster calls.
    * 
    * @deprecated Use {@link GroupMembershipListener}
    */
   @Deprecated
   public interface AsynchHAMembershipListener extends HAMembershipListener
   {
      // Nothing new
   }
   
   /** 
    * Extends HAMembershipListener to receive a specialized notification when a 
    * network-partition merge occurs. A merge occurs when two or more nodes that 
    * were unaware of each other for some reason (e.g. a crashed switch) detect 
    * each other and form a single group.
    * 
    * @deprecated Use {@link GroupMembershipListener}
    */
   @Deprecated
   public interface HAMembershipExtendedListener extends HAPartition.HAMembershipListener
   {
      /** 
       * Specialized notification issued instead of 
       * {@link #membershipChanged(List, List, List) the standard one}
       * when a network-partition merge occurs. This notification is made
       * using the thread that carries messages up from the network and so you 
       * should not execute new cluster calls using this thread. If you need to 
       * do that implement the 
       * {@link AsynchHAMembershipExtendedListener asynchronous version of this interface}.
       *
       * @param deadMembers A list of nodes that have died since the previous view
       * @param newMembers A list of nodes that have joined the partition since the previous view
       * @param allMembers A list of nodes that built the current view
       * @param originatingGroups A list of nodes that were previously partioned and that are now merged
       */      
      public void membershipChangedDuringMerge(Vector<ClusterNode> deadMembers, Vector<ClusterNode> newMembers,
            Vector<ClusterNode> allMembers, Vector<List<ClusterNode>> originatingGroups);
   }

   /** 
    * A tagging interface for HAMembershipExtendedListener callbacks that will
    * be performed in a thread separate from the thread that carries messages 
    * up from the network. The ordering of view changes is preserved, but 
    * listeners are free to execute cluster calls.
    * 
    * @deprecated Use {@link GroupMembershipListener}
    */
   @Deprecated
   public interface AsynchHAMembershipExtendedListener extends HAMembershipExtendedListener
   {
      // Nothing new
   }

   /**
    * Subscribes to receive {@link HAMembershipListener} events.
    * @param listener The membership listener object
    * 
    * @deprecated Use {@link #registerGroupMembershipListener(GroupMembershipListener)}
    */   
   @Deprecated
   public void registerMembershipListener(HAMembershipListener listener);
   
   /**
    * Unsubscribes from receiving {@link HAMembershipListener} events.
    * @param listener The listener wishing to unsubscribe
    * 
    * @deprecated Use {@link #unregisterGroupMembershipListener(GroupMembershipListener)}
    */   
   @Deprecated
   public void unregisterMembershipListener(HAMembershipListener listener);
   
   /**
    * Returns whether this partition will synchronously notify any 
    * HAMembershipListeners of membership changes using the calling thread
    * from the underlying <code>ClusterPartition</code>.
    * 
    * @return <code>true</code> if registered listeners that don't implement
    *         <code>AsynchHAMembershipExtendedListener</code> or
    *         <code>AsynchHAMembershipListener</code> will be notified
    *         synchronously of membership changes; <code>false</code> if
    *         those listeners will be notified asynchronously.  Default
    *         is <code>false</code>.
    * 
    * @deprecated Use {@link GroupMembershipListener} API; synchronous notifications
    *             will eventually not be allowed
    */   
   @Deprecated
   public boolean getAllowSynchronousMembershipNotifications();

   /**
    * Return the list of member nodes that built the current view i.e., the current partition.
    * @return An array of Strings containing the node names
    */   
   public Vector<String> getCurrentView ();
}