/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.ha.framework.server;

import java.util.concurrent.atomic.AtomicInteger;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheManager;
import org.jboss.cache.CacheStatus;
import org.jboss.ha.core.framework.server.ChannelSource;
import org.jboss.ha.framework.server.spi.HAPartitionCacheHandler;
import org.jgroups.Channel;
import org.jgroups.ChannelFactory;

/**
 * Standard implementation of HAPartitionCacheHandler
 * 
 * @author Brian Stansberry
 */
public class HAPartitionCacheHandlerImpl implements HAPartitionCacheHandler, ChannelSource
{
   private CacheManager cacheManager;
   private String cacheConfigName;
   private Cache<Object, Object> cache;
   private AtomicInteger acquireCount = new AtomicInteger();
   
   // CacheHandler ------------------------------------------------------------

   public synchronized void acquireCache() throws Exception
   {
      if (cache == null)
      {
         validateState();
         
         cache = cacheManager.getCache(cacheConfigName, true);
      }
      acquireCount.incrementAndGet();
   }

   public synchronized void startCache() throws Exception
   {
      if (cache == null)
      {
         throw new IllegalStateException("Must acquire cache before starting it");
      }
      
      if (cache.getCacheStatus() != CacheStatus.STARTED)
         cache.start();
   }

   public synchronized void releaseCache() throws Exception
   {
      if (cache != null)
      {
         validateState();
      
         if (acquireCount.decrementAndGet() == 0)
         {
            cacheManager.releaseCache(cacheConfigName);
            cache = null;
         }
      }
   }

   public synchronized String getCacheConfigName()
   {
      return cacheConfigName;
   }

   public synchronized String getChannelStackName()
   {
      if (cache == null)
      {
         throw new IllegalStateException("Must acquire cache before getting channel stack name");
      }
      return cache.getConfiguration().getMultiplexerStack();
   }

   public synchronized ChannelFactory getCacheChannelFactory()
   {
      if (cache == null)
      {
         throw new IllegalStateException("Must acquire cache before getting cache channel factory");
      }
      return cache.getConfiguration().getRuntimeConfig().getMuxChannelFactory();
   }
   
   public synchronized Channel getChannel()
   {
      if (cache == null)
      {
         throw new IllegalStateException("Must acquire cache before getting cache channel");
      }
      return cache.getConfiguration().getRuntimeConfig().getChannel();
   }
   
   // Properties -------------------------------------------------------------

   public synchronized void setCacheConfigName(String cacheConfigName)
   {
      this.cacheConfigName = cacheConfigName;
   }

   public synchronized CacheManager getCacheManager()
   {
      return cacheManager;
   }

   public synchronized void setCacheManager(CacheManager cacheManager)
   {
      this.cacheManager = cacheManager;
   }
   
   public synchronized Cache<Object, Object> getCache()
   {
      return cache;
   }
   
   // Public ------------------------------------------------------------------
   
   public synchronized void start() throws Exception
   {
   }
   
   public synchronized void stop() throws Exception
   {
   }
   
   // Private -----------------------------------------------------------------

   private void validateState()
   {
      if (cacheManager == null)
      {
         throw new IllegalStateException("Must set cacheManager");
      }
      
      if (cacheConfigName == null)
      {
         throw new IllegalStateException("Must set cacheConfigName");
      }
   }
}
