/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat, Inc. and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.tomcat.service.session.distributedcache.impl;

import java.util.ArrayList;
import java.util.List;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheStatus;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.buddyreplication.NextMemberBuddyLocatorConfig;
import org.jboss.cache.config.BuddyReplicationConfig;
import org.jboss.cache.config.CacheLoaderConfig;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.CacheLoaderConfig.IndividualCacheLoaderConfig;
import org.jboss.cache.config.Configuration.CacheMode;
import org.jboss.cache.config.Configuration.NodeLockingScheme;
import org.jboss.cache.loader.FileCacheLoaderConfig;
import org.jboss.cache.lock.IsolationLevel;
import org.jboss.cache.transaction.BatchModeTransactionManagerLookup;
import org.jboss.logging.Logger;
import org.jboss.web.tomcat.service.session.distributedcache.spi.ClusteringNotSupportedException;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.OutgoingDistributableSessionData;
import org.jboss.web.tomcat.service.session.distributedcache.spi.TestDistributedCacheManagerFactory;

/**
 *
 *
 * @author Brian Stansberry
 * 
 * @version $Revision$
 */
public class TestDistributedCacheManagerFactoryImpl implements TestDistributedCacheManagerFactory
{
   private static final Logger log = Logger.getLogger(TestDistributedCacheManagerFactoryImpl.class);
   
   private DistributedCacheManagerFactoryImpl delegate = new DistributedCacheManagerFactoryImpl();

   public void configure(boolean local, String passivationDir, boolean totalReplication, boolean marshalling,
         boolean purgeCacheLoader)
   {
      Cache pc = createCache(local, passivationDir, totalReplication, marshalling, purgeCacheLoader);
      delegate.setPlainCache(pc);
   }

   @SuppressWarnings("unchecked")
   public <T extends OutgoingDistributableSessionData> DistributedCacheManager<T> getDistributedCacheManager(
         LocalDistributableSessionManager localManager) throws ClusteringNotSupportedException
   {
      return delegate.getDistributedCacheManager(localManager);
   }  

   public void cleanup(boolean doRemove)
   {
      Cache cache = delegate.getPlainCache();
      if (cache == null)
         return;
      
      if (doRemove && CacheStatus.STARTED.equals(cache.getCacheStatus()))
      {
         // Try to clean up so we avoid loading sessions 
         // from storage in later tests
         try
         {
            log.info("Removing /JSESSION from " + cache.getLocalAddress());
            cache.removeNode(Fqn.fromString("/JSESSION"));
         }
         catch (Exception e)
         {
            log.error("Cache " + cache + ": " + e.getMessage(), e);
         }
      }
      
      try
      {
         cache.stop();
         cache.destroy();
      }
      catch (Exception e)
      {
         log.error("Cache " + cache + ": " + e.getMessage(), e);
      }
      
   } 
   
   

   
   private static Cache createCache(boolean local, String passivationDir,
         boolean totalReplication, boolean marshalling, boolean purgeCacheLoader)
   {
      Configuration cfg = getConfiguration(local, passivationDir, totalReplication, marshalling, purgeCacheLoader);
      return DefaultCacheFactory.getInstance().createCache(cfg, true);
   }
   
   private static Configuration getConfiguration(boolean local, String passivationDir, 
         boolean totalReplication, boolean marshalling, boolean purgeCacheLoader)
   {
      Configuration config = new Configuration();
      config.setClusterName("Tomcat-TestCluster");
      config.setTransactionManagerLookupClass(BatchModeTransactionManagerLookup.class.getName());
      config.setNodeLockingScheme(NodeLockingScheme.PESSIMISTIC);
      config.setIsolationLevel(IsolationLevel.REPEATABLE_READ);
      config.setSyncReplTimeout(20000);
      config.setLockAcquisitionTimeout(15000);
      
      config.setCacheMode(local ? CacheMode.LOCAL : CacheMode.REPL_SYNC);
      
      config.setUseRegionBasedMarshalling(marshalling);
      config.setInactiveOnStartup(marshalling);    
      
      // No async marshalling or notifications
      config.setSerializationExecutorPoolSize(0);
      config.setListenerAsyncPoolSize(0);  
      
      // Block for commits -- no races between test driver and replication
      config.setSyncCommitPhase(true);
      config.setSyncRollbackPhase(true);
      
      if (passivationDir != null)
      {
         CacheLoaderConfig clc = new CacheLoaderConfig();
         clc.setPassivation(true);
         clc.setShared(false);
         FileCacheLoaderConfig fclc = new FileCacheLoaderConfig();
         fclc.setLocation(passivationDir);
         fclc.setFetchPersistentState(true);
         fclc.setPurgeOnStartup(purgeCacheLoader);
         fclc.setAsync(false);
         fclc.setIgnoreModifications(false);
         ArrayList<IndividualCacheLoaderConfig> iclcs = new ArrayList<IndividualCacheLoaderConfig>();
         iclcs.add(fclc);
         clc.setIndividualCacheLoaderConfigs(iclcs);
         
         config.setCacheLoaderConfig(clc);
      }
      
      if (!local && !totalReplication)
      {
         BuddyReplicationConfig brc = new BuddyReplicationConfig();
         brc.setEnabled(true);
         NextMemberBuddyLocatorConfig blc = new NextMemberBuddyLocatorConfig();
         blc.setNumBuddies(1);
         blc.setIgnoreColocatedBuddies(true);
         brc.setBuddyLocatorConfig(blc);
         
         brc.setBuddyPoolName("default");
         brc.setBuddyCommunicationTimeout(20000);
         brc.setAutoDataGravitation(false);
         brc.setDataGravitationRemoveOnFind(true);
         brc.setDataGravitationSearchBackupTrees(true);
         
         config.setBuddyReplicationConfig(brc);
      }
      
      return config;
   }

   public Object getLocalAddress()
   {
      Cache pc = delegate.getPlainCache();
      return pc == null ? null : pc.getLocalAddress();
   }

   public List<Object> getMembers()
   {
      Cache pc = delegate.getPlainCache();
      return pc == null ? null : new ArrayList<Object>(pc.getMembers());
   }

}
