/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.deployers.vfs.plugins.annotations;

import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.mcann.repository.AbstractSettings;
import org.jboss.mcann.repository.AbstractConfiguration;
import org.jboss.mcann.scanner.DefaultAnnotationScanner;

/**
 * Filtered annotation environment deployer.
 *
 * It first checks if there are some filters present
 * in deployment unit as attachment,
 * else falls back to deployers filters.
 *
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
public class FilteredAnnotationRepositoryDeployer extends AnnotationRepositoryDeployer
{
   private ResourceFilter resourceFilter;
   private ResourceFilter recurseFilter;

   public FilteredAnnotationRepositoryDeployer()
   {
      addInput(ResourceFilter.class.getName() + ".resource");
      addInput(ResourceFilter.class.getName() + ".recurse");
   }

   /**
    * Try attachment first, then default value.
    *
    * @param <T> the expected class type
    * @param unit the deployment unit
    * @param expectedClass the expected class
    * @param suffix the suffix
    * @param defaultValue the default value
    * @return attchment or default value
    */
   protected <T> T getAttachment(DeploymentUnit unit, Class<T> expectedClass, String suffix, T defaultValue)
   {
      String name = expectedClass.getName() + (suffix != null ? ("." + suffix) : "");
      T result = unit.getAttachment(name, expectedClass);
      if (result == null)
         result = defaultValue;
      return result;
   }

   @Override
   protected AbstractConfiguration createConfiguration(VFSDeploymentUnit unit)
   {
      return getAttachment(unit, AbstractConfiguration.class, null, super.createConfiguration(unit)); 
   }

   /**
    * We look for filter attachments:
    * * org.jboss.classloading.spi.visitor.ResourceFilter.resource - plain resource filter
    * * org.jboss.classloading.spi.visitor.ResourceFilter.recurse  - recurse resource filter
    *
    * @param unit the deployment unit
    * @param scanner the annotation scanner
    * @param settings the settings
    */
   @Override
   protected void configureScanner(VFSDeploymentUnit unit, DefaultAnnotationScanner scanner, AbstractSettings settings)
   {
      super.configureScanner(unit, scanner, settings);

      ResourceFilter filter = getAttachment(unit, ResourceFilter.class, "resource", resourceFilter);
      settings.setResourceFilter(filter);

      ResourceFilter recurse = getAttachment(unit, ResourceFilter.class, "recurse", recurseFilter);
      scanner.setRecurseFilter(recurse);
   }

   /**
    * Set resource filter.
    *
     * @param resourceFilter the resource filter
    */
   public void setResourceFilter(ResourceFilter resourceFilter)
   {
      this.resourceFilter = resourceFilter;
   }

   /**
    * Set recurse filter.
    *
    * @param recurseFilter the recurse filter
    */
   public void setRecurseFilter(ResourceFilter recurseFilter)
   {
      this.recurseFilter = recurseFilter;
   }
}