/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.deployers;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.helpers.AbstractComponentDeployer;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeploymentVisitor;
import org.jboss.deployers.spi.deployer.helpers.DeploymentVisitor;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.ejb3.common.deployers.spi.AttachmentNames;
import org.jboss.logging.Logger;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeanMetaData;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeansMetaData;
import org.jboss.metadata.ejb.jboss.JBossMetaData;

import java.util.ArrayList;
import java.util.List;

/**
 * Create components out of EJB 3 JBossMetaData.
 * 
 * @author <a href="mailto:cdewolf@redhat.com">Carlo de Wolf</a>
 * @version $Revision: $
 */
public class EJBsDeployer extends AbstractComponentDeployer<JBossMetaData, JBossEnterpriseBeanMetaData>
{
   private static final Logger log = Logger.getLogger(EJBsDeployer.class);
   
   private static final JBossDeploymentVisitor deploymentVisitor = new JBossDeploymentVisitor();

   private String attachmentName = AttachmentNames.PROCESSED_METADATA;
   
   /**
    * 
    */
   public EJBsDeployer()
   {
      // Since we don't set component visitor
      addInput(attachmentName);
      setOutput(deploymentVisitor.getComponentType());
      
      setDeploymentVisitor(deploymentVisitor);
   }
   
   private static class JBossDeploymentVisitor extends AbstractDeploymentVisitor<JBossEnterpriseBeanMetaData, JBossMetaData>
   {
      @Override
      public Class<JBossEnterpriseBeanMetaData> getComponentType()
      {
         return JBossEnterpriseBeanMetaData.class;
      }

      @Override
      protected List<? extends JBossEnterpriseBeanMetaData> getComponents(JBossMetaData deployment)
      {
         if(deployment == null || !deployment.isEJB3x())
            return null;
         JBossEnterpriseBeansMetaData enterpriseBeans = deployment.getEnterpriseBeans();
         if(enterpriseBeans == null)
         {
            log.warn("EJBTHREE-2095: did not find any beans in " + deployment + " which is a violation of the xsd");
            return null;
         }
         return new ArrayList<JBossEnterpriseBeanMetaData>(enterpriseBeans);
      }

      @Override
      protected String getComponentName(JBossEnterpriseBeanMetaData attachment)
      {
         return JBossEnterpriseBeanMetaData.class.getName() + "." + attachment.getEjbName();
      }

      public Class<JBossMetaData> getVisitorType()
      {
         return JBossMetaData.class;
      }
   }
   
   @Override
   protected <U> void deploy(DeploymentUnit unit, DeploymentVisitor<U> visitor) throws DeploymentException
   {
      // I only want a single attachment deployed
      
      U deployment = unit.getAttachment(attachmentName, visitor.getVisitorType());
      try
      {
         visitor.deploy(unit, deployment);
      }
      catch(Throwable t)
      {
         throw DeploymentException.rethrowAsDeploymentException("Error deploying: " + unit.getName(), t);
      }
   }
   
   @Override
   protected <U> void undeploy(DeploymentUnit unit, DeploymentVisitor<U> visitor)
   {
      if(visitor == null)
         return;
      
      // I only want a single attachment undeployed

      U deployment = unit.getAttachment(attachmentName, visitor.getVisitorType());
      visitor.undeploy(unit, deployment);
   }
}
