/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.ejb3.metadata.deployers;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.ejb3.deployers.Ejb3MetadataProcessingDeployer;
import org.jboss.logging.Logger;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeanMetaData;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeansMetaData;
import org.jboss.metadata.ejb.jboss.JBossMetaData;
import org.jboss.metadata.ejb.jboss.JBossSessionBeanMetaData;
import org.jboss.metadata.ejb.jboss.jndi.resolver.impl.JNDIPolicyBasedJNDINameResolverFactory;
import org.jboss.metadata.ejb.jboss.jndi.resolver.spi.SessionBeanJNDINameResolver;
import org.jboss.metadata.ejb.jboss.jndipolicy.plugins.DefaultJNDIBindingPolicyFactory;
import org.jboss.metadata.ejb.jboss.jndipolicy.spi.DefaultJndiBindingPolicy;

/**
 * JBossMetaDataResolvedJNDINameDeployer
 * 
 * Responsible for setting the resolved JNDI names on the metadata, so 
 * that the code relying on calls like metadata.getJNDIName will 
 * get the resolved jndi name.
 * 
 * Note that this deployer is here to provide backward compatibility
 * and any new code which expects resolved JNDI names, is expected
 * to use the JNDI name resolvers (ex: an implementation of {@link SessionBeanJNDINameResolver})  
 *
 * Calls to 
 * <ul>
 *  <li>{@link JBossSessionBeanMetaData#getJndiName()}</li>
 *  <li>{@link JBossSessionBeanMetaData#getLocalJndiName()} </li>
 *  <li> {@link JBossSessionBeanMetaData#getLocalHomeJndiName()} </li>
 *  <li> {@link JBossSessionBeanMetaData#getHomeJndiName()} </li>
 * </ul>
 * will return the corresponding resolved jndi names, after this deployer processes the metadata 
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class JBossMetaDataResolvedJNDINameDeployer extends AbstractDeployer
{

   /**
    * Logger
    */
   private static final Logger logger = Logger.getLogger(JBossMetaDataResolvedJNDINameDeployer.class);

   /**
    * Deployer Output flag signaling that this deployer has been run
    */
   public static final String OUTPUT = JBossMetaDataResolvedJNDINameDeployer.class.getSimpleName();

   /**
    * Constructor; sets deployment stage and requisite input/output chain
    */
   public JBossMetaDataResolvedJNDINameDeployer()
   {
      // Set the Stage to post-CL
      this.setStage(DeploymentStages.PRE_REAL);

      // Input is the JBossMetaData post-merge
      this.addInput(Ejb3MetadataProcessingDeployer.OUTPUT);

      // Output is a flag upon which other deployers may rely
      this.addOutput(OUTPUT);
   }

   /**
    * @see org.jboss.deployers.spi.deployer.Deployer#deploy(org.jboss.deployers.structure.spi.DeploymentUnit)
    */
   public void deploy(DeploymentUnit du) throws DeploymentException
   {
      // Obtain the Merged Metadata
      JBossMetaData md = du.getAttachment(Ejb3MetadataProcessingDeployer.OUTPUT, JBossMetaData.class);

      // If metadata's not present as an attachment, return
      if (md == null)
      {
         return;
      }

      // If this is not an EJB3 Deployment, return
      if (!md.isEJB3x())
      {
         return;
      }

      // Decorate
      logger.debug("Decorating EJB3 beans in unit " + du + " with JNDI Policy");
      JBossEnterpriseBeansMetaData beans = md.getEnterpriseBeans();
      if (beans == null)
      {
         return;
      }
      DefaultJndiBindingPolicy jndiBindingPolicy = DefaultJNDIBindingPolicyFactory.getDefaultJNDIBindingPolicy();
      for (JBossEnterpriseBeanMetaData bean : beans)
      {
         // let's only do this for SLSB, SFSB and @Service
         // Ignore Entity beans or MDB
         if (bean.isSession() || bean.isService())
         {
            JBossSessionBeanMetaData sessionBean = (JBossSessionBeanMetaData) bean;
            SessionBeanJNDINameResolver sessionBeanJNDINameResolver = JNDIPolicyBasedJNDINameResolverFactory.getJNDINameResolver(sessionBean, jndiBindingPolicy);
            // set jndi-name (default remote business jndi-name)
            String defaultBusinessRemoteJNDIName = sessionBeanJNDINameResolver.resolveRemoteBusinessDefaultJNDIName(sessionBean);
            if (defaultBusinessRemoteJNDIName != null)
            {
               sessionBean.setJndiName(defaultBusinessRemoteJNDIName);
            }
            // set local-jndi-name
            String defaultBusinessLocalJNDIName = sessionBeanJNDINameResolver.resolveLocalBusinessDefaultJNDIName(sessionBean);
            if (defaultBusinessLocalJNDIName != null)
            {
               sessionBean.setLocalJndiName(defaultBusinessLocalJNDIName);
            }
            // set local-home-jndi-name
            String localHomeJNDIName = sessionBeanJNDINameResolver.resolveLocalHomeJNDIName(sessionBean);
            if (localHomeJNDIName != null)
            {
               sessionBean.setLocalHomeJndiName(localHomeJNDIName);
            }
            // set remote-home-jndi-name
            String remoteHomeJNDIName = sessionBeanJNDINameResolver.resolveRemoteHomeJNDIName(sessionBean);
            if (remoteHomeJNDIName != null)
            {
               sessionBean.setHomeJndiName(remoteHomeJNDIName);
            }
         }
      }
      // Output
      du.addAttachment(OUTPUT, true, Boolean.class);
   }
}
