/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.elasticsearch.river.remote.sitemap;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.zip.GZIPInputStream;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.apache.commons.io.input.BOMInputStream;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.Loggers;
import org.jboss.elasticsearch.river.remote.sitemap.AbstractSiteMap;
import org.jboss.elasticsearch.river.remote.sitemap.SiteMap;
import org.jboss.elasticsearch.river.remote.sitemap.SiteMapIndex;
import org.jboss.elasticsearch.river.remote.sitemap.SiteMapURL;
import org.jboss.elasticsearch.river.remote.sitemap.UnknownFormatException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

public class SiteMapParser {
    private static final ESLogger logger = Loggers.getLogger(SiteMapParser.class);
    private static final int MAX_URLS = 50000;
    public static int MAX_BYTES_ALLOWED = 0xA00000;
    private boolean strict;

    public SiteMapParser() {
        this(true);
    }

    public SiteMapParser(boolean strict) {
        this.strict = strict;
    }

    public boolean isStrict() {
        return this.strict;
    }

    public AbstractSiteMap parseSiteMap(String contentType, byte[] content, URL url) throws UnknownFormatException, IOException {
        if (url.getPath().endsWith(".xml") || contentType.contains("text/xml") || contentType.contains("application/xml") || contentType.contains("application/x-xml") || contentType.contains("application/atom+xml") || contentType.contains("application/rss+xml")) {
            return this.processXml(url, content);
        }
        if (url.getPath().endsWith(".txt") || contentType.contains("text/plain")) {
            return this.processText(content, url.toString());
        }
        if (url.getPath().endsWith(".gz") || contentType.contains("application/gzip") || contentType.contains("application/x-gzip") || contentType.contains("application/x-gunzip") || contentType.contains("application/gzipped") || contentType.contains("application/gzip-compressed") || contentType.contains("application/x-compress") || contentType.contains("gzip/document") || contentType.contains("application/octet-stream")) {
            return this.processGzip(url, content);
        }
        throw new UnknownFormatException("Unknown format " + contentType + " at " + url);
    }

    private AbstractSiteMap processXml(URL sitemapUrl, byte[] xmlContent) throws UnknownFormatException {
        BOMInputStream bomIs = new BOMInputStream((InputStream)new ByteArrayInputStream(xmlContent));
        InputSource is = new InputSource();
        is.setCharacterStream(new BufferedReader(new InputStreamReader((InputStream)bomIs)));
        return this.processXml(sitemapUrl, is);
    }

    private SiteMap processText(byte[] content, String sitemapUrl) throws IOException {
        String line;
        logger.debug("Processing textual Sitemap", new Object[0]);
        SiteMap textSiteMap = new SiteMap(sitemapUrl);
        textSiteMap.setType(AbstractSiteMap.SitemapType.TEXT);
        BOMInputStream bomIs = new BOMInputStream((InputStream)new ByteArrayInputStream(content));
        BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)bomIs));
        int i = 1;
        while ((line = reader.readLine()) != null) {
            if (line.length() <= 0 || i > 50000) continue;
            try {
                URL url = new URL(line);
                boolean valid = SiteMapParser.urlIsLegal(textSiteMap.getBaseUrl(), url.toString());
                if (!valid && this.strict) continue;
                if (logger.isDebugEnabled()) {
                    StringBuffer sb = new StringBuffer("  ");
                    sb.append(i).append(". ").append(url);
                    logger.debug(sb.toString(), new Object[0]);
                }
                ++i;
                SiteMapURL surl = new SiteMapURL(url, valid);
                textSiteMap.addSiteMapUrl(surl);
            }
            catch (MalformedURLException e) {
                logger.debug("Bad URL [" + line + "].", new Object[0]);
            }
        }
        textSiteMap.setProcessed(true);
        return textSiteMap;
    }

    private AbstractSiteMap processGzip(URL url, byte[] response) throws MalformedURLException, IOException, UnknownFormatException {
        logger.debug("Processing gzip", new Object[0]);
        ByteArrayInputStream is = new ByteArrayInputStream(response);
        String xmlUrl = url.toString().replaceFirst("\\.gz$", "");
        logger.debug("XML url = " + xmlUrl, new Object[0]);
        BOMInputStream decompressed = new BOMInputStream((InputStream)new GZIPInputStream(is));
        InputSource in = new InputSource((InputStream)decompressed);
        in.setSystemId(xmlUrl);
        AbstractSiteMap smi = this.processXml(url, in);
        decompressed.close();
        return smi;
    }

    private AbstractSiteMap processXml(URL sitemapUrl, InputSource is) throws UnknownFormatException {
        Document doc = null;
        try {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            db.setErrorHandler(new ErrorHandler(){

                @Override
                public void warning(SAXParseException exception) throws SAXException {
                    logger.warn("Sitemap XML warning: " + exception.getMessage(), new Object[0]);
                }

                @Override
                public void fatalError(SAXParseException exception) throws SAXException {
                    logger.warn("Sitemap XML fatalError: " + exception.getMessage(), new Object[0]);
                }

                @Override
                public void error(SAXParseException exception) throws SAXException {
                    logger.warn("Sitemap XML error: " + exception.getMessage(), new Object[0]);
                }
            });
            doc = db.parse(is);
        }
        catch (Exception e) {
            throw new UnknownFormatException("Error parsing XML for " + sitemapUrl);
        }
        NodeList nodeList = doc.getElementsByTagName("sitemapindex");
        if (nodeList.getLength() > 0) {
            nodeList = doc.getElementsByTagName("sitemap");
            return this.parseSitemapIndex(sitemapUrl, nodeList);
        }
        if (doc.getElementsByTagName("urlset").getLength() > 0) {
            return this.parseXmlSitemap(sitemapUrl, doc);
        }
        if (doc.getElementsByTagName("link").getLength() > 0) {
            return this.parseSyndicationFormat(sitemapUrl, doc);
        }
        throw new UnknownFormatException("Unknown XML format for " + sitemapUrl);
    }

    private SiteMap parseXmlSitemap(URL sitemapUrl, Document doc) {
        SiteMap sitemap = new SiteMap(sitemapUrl);
        sitemap.setType(AbstractSiteMap.SitemapType.XML);
        NodeList list = doc.getElementsByTagName("url");
        for (int i = 0; i < list.getLength(); ++i) {
            Node n = list.item(i);
            if (n.getNodeType() != 1) continue;
            Element elem = (Element)n;
            String loc = this.getElementValue(elem, "loc");
            URL url = null;
            try {
                url = new URL(loc);
                String lastMod = this.getElementValue(elem, "lastmod");
                String changeFreq = this.getElementValue(elem, "changefreq");
                String priority = this.getElementValue(elem, "priority");
                boolean valid = SiteMapParser.urlIsLegal(sitemap.getBaseUrl(), url.toString());
                if (!valid && this.strict) continue;
                SiteMapURL sUrl = new SiteMapURL(url.toString(), lastMod, changeFreq, priority, valid);
                sitemap.addSiteMapUrl(sUrl);
                if (!logger.isDebugEnabled()) continue;
                StringBuffer sb = new StringBuffer("  ");
                sb.append(i + 1).append(". ").append(sUrl);
                logger.debug(sb.toString(), new Object[0]);
                continue;
            }
            catch (MalformedURLException e) {
                logger.debug("Bad url: [" + loc + "]", new Object[0]);
            }
        }
        sitemap.setProcessed(true);
        return sitemap;
    }

    private SiteMapIndex parseSitemapIndex(URL url, NodeList nodeList) {
        logger.debug("Parsing Sitemap Index", new Object[0]);
        SiteMapIndex sitemapIndex = new SiteMapIndex(url);
        sitemapIndex.setType(AbstractSiteMap.SitemapType.INDEX);
        for (int i = 0; i < nodeList.getLength() && i < 50000; ++i) {
            Node firstNode = nodeList.item(i);
            URL sitemapUrl = null;
            Date lastModified = null;
            if (firstNode.getNodeType() != 1) continue;
            Element elem = (Element)firstNode;
            String loc = this.getElementValue(elem, "loc");
            if (loc == null) {
                loc = elem.getTextContent().trim();
            }
            try {
                sitemapUrl = new URL(loc);
                String lastmod = this.getElementValue(elem, "lastmod");
                lastModified = SiteMap.convertToDate(lastmod);
                SiteMap s = new SiteMap(sitemapUrl, lastModified);
                sitemapIndex.addSitemap(s);
                if (!logger.isDebugEnabled()) continue;
                StringBuffer sb = new StringBuffer("  ");
                sb.append(i + 1).append(". ").append(s);
                logger.debug(sb.toString(), new Object[0]);
                continue;
            }
            catch (MalformedURLException e) {
                logger.debug("Bad url: [" + loc + "]", new Object[0]);
            }
        }
        sitemapIndex.setProcessed(true);
        return sitemapIndex;
    }

    private SiteMap parseSyndicationFormat(URL sitemapUrl, Document doc) throws UnknownFormatException {
        SiteMap sitemap = new SiteMap(sitemapUrl);
        NodeList list = doc.getElementsByTagName("feed");
        if (list.getLength() > 0) {
            this.parseAtom(sitemap, (Element)list.item(0), doc);
            sitemap.setProcessed(true);
            return sitemap;
        }
        list = doc.getElementsByTagName("rss");
        if (list.getLength() > 0) {
            this.parseRSS(sitemap, doc);
            sitemap.setProcessed(true);
            return sitemap;
        }
        throw new UnknownFormatException("Unknown syndication format at " + sitemapUrl);
    }

    private void parseAtom(SiteMap sitemap, Element elem, Document doc) {
        logger.debug("Parsing Atom XML", new Object[0]);
        sitemap.setType(AbstractSiteMap.SitemapType.ATOM);
        String lastMod = this.getElementValue(elem, "modified");
        logger.debug("lastMod=" + lastMod, new Object[0]);
        NodeList list = doc.getElementsByTagName("entry");
        for (int i = 0; i < list.getLength() && i < 50000; ++i) {
            Node n = list.item(i);
            if (n.getNodeType() != 1) continue;
            elem = (Element)n;
            String href = this.getElementAttributeValue(elem, "link", "href");
            logger.debug("href=" + href, new Object[0]);
            URL url = null;
            try {
                url = new URL(href);
                boolean valid = SiteMapParser.urlIsLegal(sitemap.getBaseUrl(), url.toString());
                if (!valid && this.strict) continue;
                SiteMapURL sUrl = new SiteMapURL(url.toString(), lastMod, null, null, valid);
                sitemap.addSiteMapUrl(sUrl);
                if (!logger.isDebugEnabled()) continue;
                StringBuffer sb = new StringBuffer("  ");
                sb.append(i + 1).append(". ").append(sUrl);
                logger.debug(sb.toString(), new Object[0]);
                continue;
            }
            catch (MalformedURLException e) {
                logger.debug("Bad url: [" + href + "]", new Object[0]);
            }
        }
    }

    private void parseRSS(SiteMap sitemap, Document doc) {
        logger.debug("Parsing RSS doc", new Object[0]);
        sitemap.setType(AbstractSiteMap.SitemapType.RSS);
        NodeList list = doc.getElementsByTagName("channel");
        Element elem = (Element)list.item(0);
        String lastMod = this.getElementValue(elem, "pubDate");
        logger.debug("lastMod=" + lastMod, new Object[0]);
        list = doc.getElementsByTagName("item");
        for (int i = 0; i < list.getLength() && i < 50000; ++i) {
            Node n = list.item(i);
            if (n.getNodeType() != 1) continue;
            elem = (Element)n;
            String link = this.getElementValue(elem, "link");
            logger.debug("link=" + link, new Object[0]);
            try {
                URL url = new URL(link);
                boolean valid = SiteMapParser.urlIsLegal(sitemap.getBaseUrl(), url.toString());
                if (!valid && this.strict) continue;
                SiteMapURL sUrl = new SiteMapURL(url.toString(), lastMod, null, null, valid);
                sitemap.addSiteMapUrl(sUrl);
                if (!logger.isDebugEnabled()) continue;
                StringBuffer sb = new StringBuffer("  ");
                sb.append(i + 1).append(". ").append(sUrl);
                logger.debug(sb.toString(), new Object[0]);
                continue;
            }
            catch (MalformedURLException e) {
                logger.debug("Bad url: [" + link + "]", new Object[0]);
            }
        }
    }

    private String getElementValue(Element elem, String elementName) {
        NodeList children;
        NodeList list = elem.getElementsByTagName(elementName);
        Element e = (Element)list.item(0);
        if (e != null && (children = e.getChildNodes()).item(0) != null) {
            return children.item(0).getNodeValue().trim();
        }
        return null;
    }

    private String getElementAttributeValue(Element elem, String elementName, String attributeName) {
        NodeList list = elem.getElementsByTagName(elementName);
        Element e = (Element)list.item(0);
        if (e != null) {
            return e.getAttribute(attributeName);
        }
        return null;
    }

    protected static boolean urlIsLegal(String sitemapBaseUrl, String testUrl) {
        boolean ret = false;
        if (sitemapBaseUrl != null && sitemapBaseUrl.length() <= testUrl.length()) {
            String u = testUrl.substring(0, sitemapBaseUrl.length()).toLowerCase();
            ret = sitemapBaseUrl.toLowerCase().equals(u);
        }
        if (logger.isTraceEnabled()) {
            StringBuffer sb = new StringBuffer("urlIsLegal: ");
            sb.append(sitemapBaseUrl).append(" <= ").append(testUrl);
            sb.append(" ? ").append(ret);
            logger.trace(sb.toString(), new Object[0]);
        }
        return ret;
    }
}

