/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.bindings.tomcat.idp;

import java.security.PrivateKey;
import java.security.PublicKey;

import org.apache.catalina.LifecycleException;
import org.apache.catalina.connector.Request;
import org.apache.log4j.Logger;
import org.jboss.identity.federation.bindings.config.KeyProvider;
import org.jboss.identity.federation.bindings.interfaces.TrustKeyManager;
import org.jboss.identity.federation.bindings.util.RedirectBindingSignatureUtil;
import org.jboss.identity.federation.core.saml.v2.util.SignatureUtil;


/**
 * Valve at the Identity Provider that supports
 * SAML2 HTTP/Redirect binding with digital signature support
 * @author Anil.Saldhana@redhat.com
 * @since Jan 14, 2009
 */
public class IDPRedirectWithSignatureValve extends IDPRedirectValve
{   
   private static Logger log = Logger.getLogger(IDPRedirectWithSignatureValve.class); 
   
   private boolean ignoreSignature = false;
   
   private TrustKeyManager keyManager; 
   
   public IDPRedirectWithSignatureValve()
   {
      super();  
   }
   
   public void setIgnoreSignature(String val)
   {
     if(val != null && val.length() > 0)
        this.ignoreSignature = Boolean.valueOf(val); 
   } 
   
   @Override
   public void start() throws LifecycleException
   { 
      super.start();
      KeyProvider keyProvider = this.idpConfiguration.getKeyProvider();
      try
      {
         ClassLoader tcl = SecurityActions.getContextClassLoader();
         String keyManagerClassName = keyProvider.getClassName();
         if(keyManagerClassName == null)
            throw new RuntimeException("KeyManager class name is null");
         
         Class<?> clazz = tcl.loadClass(keyManagerClassName);
         this.keyManager = (TrustKeyManager) clazz.newInstance();
         keyManager.setAuthProperties(keyProvider.getAuth());
         keyManager.setValidatingAlias(keyProvider.getValidatingAlias());
      }
      catch(Exception e)
      {
         log.error("Exception reading configuration:",e);
         throw new LifecycleException(e.getLocalizedMessage());
      }
      log.trace("Key Provider=" + keyProvider.getClassName());
   }   
   
   protected boolean validate(Request request) throws Exception
   {
      boolean result = super.validate(request);
      if( result == false)
         return result;
      
      if(this.ignoreSignature)
      {
         log.trace("Since signature is to be ignored, validation returns"); 
         return true;  
      }
      
      String queryString = request.getQueryString();
      //Check if there is a signature   
      byte[] sigValue = RedirectBindingSignatureUtil.getSignatureValueFromSignedURL(queryString);
      if(sigValue == null)
         return false;
      
      //Construct the url again
      String reqFromURL = RedirectBindingSignatureUtil.getTokenValue(queryString, "SAMLRequest"); 
      String relayStateFromURL = RedirectBindingSignatureUtil.getTokenValue(queryString, "RelayState");
      String sigAlgFromURL = RedirectBindingSignatureUtil.getTokenValue(queryString, "SigAlg"); 

      StringBuilder sb = new StringBuilder();
      sb.append("SAMLRequest=").append(reqFromURL);
       
      if(relayStateFromURL != null && relayStateFromURL.length() > 0)
      {
         sb.append("&RelayState=").append(relayStateFromURL);
      }
      sb.append("&SigAlg=").append(sigAlgFromURL);
      
      PublicKey validatingKey = keyManager.getValidatingKey(request.getRemoteAddr());
      boolean isValid = SignatureUtil.validate(sb.toString().getBytes("UTF-8"), sigValue, validatingKey);
      return isValid;     
   }
   
   @Override
   protected String getDestination(String urlEncodedResponse, String urlEncodedRelayState)
   {
      try
      {
         //Get the signing key  
         PrivateKey signingKey = keyManager.getSigningKey();
         StringBuffer sb = new StringBuffer();
         String url = RedirectBindingSignatureUtil.getSAMLResponseURLWithSignature(urlEncodedResponse, urlEncodedRelayState, signingKey);
         sb.append("?").append(url);
         return sb.toString();
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }
}