/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.bindings.jboss.trust;

import javax.xml.ws.handler.MessageContext;

import org.jboss.identity.federation.api.wstrust.STSConfiguration;
import org.jboss.identity.federation.api.wstrust.SecurityTokenProvider;
import org.jboss.identity.federation.api.wstrust.WSTrustException;
import org.jboss.identity.federation.api.wstrust.WSTrustRequestContext;
import org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler;
import org.jboss.identity.federation.api.wstrust.WSTrustUtil;
import org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken;
import org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityTokenResponse;
import org.jboss.identity.federation.ws.policy.AppliesTo;
import org.jboss.identity.federation.ws.trust.LifetimeType;
import org.jboss.identity.federation.ws.trust.RequestedSecurityTokenType;

/**
 * <p>
 * Default implementation of the {@code WSTrustRequestHandler} interface. It creates the request context containing the
 * original WS-Trust request as well as any information that may be relevant to the token processing, and delegates the
 * actual token handling processing to the appropriate {@code SecurityTokenProvider}.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class JBossRequestHandler implements WSTrustRequestHandler
{

   private STSConfiguration configuration;

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#initialize(org.jboss.identity.federation.api.wstrust.STSConfiguration)
    */
   public void initialize(STSConfiguration configuration)
   {
      this.configuration = configuration;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#issue(org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken,
    *      javax.xml.ws.handler.MessageContext)
    */
   public RequestSecurityTokenResponse issue(RequestSecurityToken request, MessageContext context)
         throws WSTrustException
   {
      SecurityTokenProvider provider = null;
      
      // first try to obtain the security token provider using the applies-to contents.
      AppliesTo appliesTo = request.getAppliesTo();
      if(appliesTo != null)
      {
         String serviceName = WSTrustUtil.parseAppliesTo(appliesTo);
         if(serviceName != null)
            provider = this.configuration.getProviderForService(serviceName);
      }
      // if applies-to is not available or if no provider was found for the service, use the token type.
      if(provider == null && request.getTokenType() != null)
      {
         provider = this.configuration.getProviderForTokenType(request.getTokenType().toString());
      }
      else if(appliesTo == null && request.getTokenType() == null)
         throw new WSTrustException("Either AppliesTo or TokenType must be present in a security token request");
      
      if(provider != null)
      {
         // create the request context and delegate token generation to the provider.
         WSTrustRequestContext requestContext = new WSTrustRequestContext(request);
         if(request.getLifetime() == null && this.configuration.getIssuedTokenTimeout() != 0)
         {
            // if no lifetime has been specified, use the configured timeout value.
            LifetimeType lifetime = WSTrustUtil.createDefaultLifetime(this.configuration.getIssuedTokenTimeout());
            request.setLifetime(lifetime);
         }
         provider.issueToken(requestContext);
         
         // construct the ws-trust security token response.
         RequestedSecurityTokenType requestedSecurityToken = new RequestedSecurityTokenType();
         requestedSecurityToken.setAny(requestContext.getSecurityToken());
         
         // TODO: create proof token and encrypt the token if needed
         
         RequestSecurityTokenResponse response = new RequestSecurityTokenResponse();
         if(request.getContext() != null)
            response.setContext(request.getContext());
         
         response.setTokenType(request.getTokenType());
         response.setLifetime(request.getLifetime());
         response.setAppliesTo(appliesTo);
         response.setRequestedSecurityToken(requestedSecurityToken);
         return response;
      }
      else
         throw new WSTrustException("Unable to find a token provider for the token request");
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#renew(org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken,
    *      javax.xml.ws.handler.MessageContext)
    */
   public RequestSecurityTokenResponse renew(RequestSecurityToken request, MessageContext context)
         throws WSTrustException
   {
      // TODO: implement renew logic.
      throw new UnsupportedOperationException();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#validate(org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken,
    *      javax.xml.ws.handler.MessageContext)
    */
   public RequestSecurityTokenResponse validate(RequestSecurityToken request, MessageContext context)
         throws WSTrustException
   {
      // TODO: implement validate logic.
      throw new UnsupportedOperationException();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#cancel(org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken,
    *      javax.xml.ws.handler.MessageContext)
    */
   public RequestSecurityTokenResponse cancel(RequestSecurityToken request, MessageContext context)
         throws WSTrustException
   {
      // TODO: implement cancel logic.
      throw new UnsupportedOperationException();
   }

}
