/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.bindings.jboss.trust;

import javax.annotation.Resource;
import javax.xml.transform.Source;
import javax.xml.ws.Service;
import javax.xml.ws.ServiceMode;
import javax.xml.ws.WebServiceContext;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.WebServiceProvider;

import org.jboss.identity.federation.api.wstrust.STSConfiguration;
import org.jboss.identity.federation.api.wstrust.SecurityTokenService;
import org.jboss.identity.federation.api.wstrust.WSTrustConstants;
import org.jboss.identity.federation.api.wstrust.WSTrustException;
import org.jboss.identity.federation.api.wstrust.WSTrustJAXBFactory;
import org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler;
import org.jboss.identity.federation.api.wstrust.WSTrustServiceFactory;
import org.jboss.identity.federation.api.wstrust.protocol.BaseRequestSecurityToken;
import org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken;
import org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityTokenCollection;
import org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityTokenResponse;
import org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityTokenResponseCollection;

/**
 * <p>
 * Default implementation of the {@code SecurityTokenService} interface.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
@WebServiceProvider
@ServiceMode(value = Service.Mode.PAYLOAD)
public class JBossSTS implements SecurityTokenService
{

   @Resource
   protected WebServiceContext context;

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.SecurityTokenService#invoke(javax.xml.transform.Source)
    */
   public Source invoke(Source request)
   {
      BaseRequestSecurityToken baseRequest = WSTrustJAXBFactory.getInstance().parseRequestSecurityToken(request);
      if (baseRequest instanceof RequestSecurityToken)
         return this.handleTokenRequest((RequestSecurityToken) baseRequest);
      else if (baseRequest instanceof RequestSecurityTokenCollection)
         return this.handleTokenRequestCollection((RequestSecurityTokenCollection) baseRequest);
      else
         throw new WebServiceException("Invalid security token request");
   }

   /**
    * <p>
    * Process a security token request.
    * </p>
    * 
    * @param request a {@code RequestSecurityToken} instance that contains the request information.
    * @return a {@code Source} instance representing the marshalled response.
    */
   protected Source handleTokenRequest(RequestSecurityToken request)
   {
      WSTrustRequestHandler handler = WSTrustServiceFactory.getInstance().createRequestHandler(this.getConfiguration());
      String requestType = request.getRequestType().toString();

      try
      {
         if (requestType.equals(WSTrustConstants.ISSUE_REQUEST))
            return this.marshallResponse(handler.issue(request, this.context.getMessageContext()));
         else if (requestType.equals(WSTrustConstants.RENEW_REQUEST))
            return this.marshallResponse(handler.renew(request, this.context.getMessageContext()));
         else if (requestType.equals(WSTrustConstants.CANCEL_REQUEST))
            return this.marshallResponse(handler.cancel(request, this.context.getMessageContext()));
         else if (requestType.equals(WSTrustConstants.VALIDATE_REQUEST))
            return this.marshallResponse(handler.validate(request, this.context.getMessageContext()));
         else
            throw new WSTrustException("Invalid request type: " + requestType);
      }
      catch (WSTrustException we)
      {
         throw new WebServiceException(we.getMessage(), we);
      }
   }

   /**
    * <p>
    * Process a collection of security token requests.
    * </p>
    * 
    * @param requestCollection a {@code RequestSecurityTokenCollection} containing the various requests information.
    * @return a {@code Source} instance representing the marshalled response.
    */
   protected Source handleTokenRequestCollection(RequestSecurityTokenCollection requestCollection)
   {
      // TODO: implement multiple token request handling code.
      return null;
   }

   /**
    * <p>
    * Marshalls the specified {@code RequestSecurityTokenResponse} into a {@code Source} instance.
    * </p>
    * 
    * @param response the {@code RequestSecurityTokenResponse} to be marshalled.
    * @return the resulting {@code Source} instance.
    */
   protected Source marshallResponse(RequestSecurityTokenResponse response)
   {
      // add the single response to a RequestSecurityTokenResponse collection, as per the specification.
      RequestSecurityTokenResponseCollection responseCollection = new RequestSecurityTokenResponseCollection();
      responseCollection.addRequestSecurityTokenResponse(response);
      return this.marshallResponse(responseCollection);
   }

   /**
    * <p>
    * Marshalls the specified {@code RequestSecurityTokenResponseCollection} into a {@code Source} instance.
    * </p>
    * 
    * @param responseCollection the {@code RequestSecurityTokenResponseCollection} to be marshalled.
    * @return the resulting {@code Source} instance.
    */
   protected Source marshallResponse(RequestSecurityTokenResponseCollection responseCollection)
   {
      return WSTrustJAXBFactory.getInstance().marshallRequestSecurityTokenResponse(responseCollection);
   }

   /**
    * <p>
    * Obtains the STS configuration options.
    * </p>
    * 
    * @return an instance of {@code STSConfiguration} containing the STS configuration properties.
    */
   protected STSConfiguration getConfiguration()
   {
      // TODO: create the configuration instance.
      return null;
   }
}
