/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.bindings.tomcat;

import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.UnrecoverableKeyException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.crypto.SecretKey;

import org.jboss.identity.federation.bindings.config.AuthPropertyType;
import org.jboss.identity.federation.bindings.config.KeyValueType;
import org.jboss.identity.federation.bindings.interfaces.TrustKeyManager;
import org.jboss.identity.federation.bindings.util.ValveUtil;
import org.jboss.identity.federation.bindings.util.cert.EncryptionKeyUtil;
import org.jboss.identity.federation.bindings.util.cert.KeyStoreUtil;

/**
 * KeyStore based Trust Key Manager
 * @author Anil.Saldhana@redhat.com
 * @since Jan 22, 2009
 */
public class KeyStoreKeyManager implements TrustKeyManager
{ 
   /**
    * An map of secret keys alive only for the duration of the program.
    * The keys are generated on the fly.  If you sophisticated key
    * storage, then a custom version of the {@code TrustKeyManager}
    * needs to be written that either uses a secure thumb drive or
    * a TPM module or a HSM module.
    * Also see JBoss XMLKey.
    */
   private Map<String,SecretKey> keys = new HashMap<String,SecretKey>();
   
   private HashMap<String,String> domainAliasMap = new HashMap<String,String>();  
   private HashMap<String,String> authPropsMap = new HashMap<String,String>();
   
   private KeyStore ks = null;
   
   private String keyStoreURL;
   private char[] signingKeyPass;
   private String signingAlias;
   private String keyStorePass;
   
   public static final String KEYSTORE_URL = "KeyStoreURL";
   public static final String KEYSTORE_PASS = "KeyStorePass";
   public static final String SIGNING_KEY_PASS = "SigningKeyPass";
   public static final String SIGNING_KEY_ALIAS = "SigningKeyAlias";
   
   /**
    * @see TrustKeyManager#getSigningKey()
    */
   public PrivateKey getSigningKey() throws Exception
   {
      if(ks == null)
         this.setUpKeyStore();
      
      if(ks == null)
         throw new IllegalStateException("KeyStore is null");
      return (PrivateKey) ks.getKey(this.signingAlias, this.signingKeyPass); 
   }

   /**
    * @see TrustKeyManager#getValidatingKey(String)
    */
   public PublicKey getValidatingKey(String domain) throws Exception
   {
      if(ks == null)
         this.setUpKeyStore();
      
      if(ks == null)
         throw new IllegalStateException("KeyStore is null");
      String domainAlias = this.domainAliasMap.get(domain);
      if(domainAlias == null)
         throw new IllegalStateException("Domain Alias missing for "+ domain);
      PublicKey publicKey = null;
      try
      {
         publicKey = KeyStoreUtil.getPublicKey(ks, domainAlias, this.keyStorePass.toCharArray());
      }
      catch(UnrecoverableKeyException urke)
      {
         //Try with the signing key pass
         publicKey = KeyStoreUtil.getPublicKey(ks, domainAlias, this.signingKeyPass);
      }
      return publicKey;
   }

   /**
    * @see TrustKeyManager#setAuthProperties(List)
    */
   public void setAuthProperties(List<AuthPropertyType> authList) throws Exception
   {
      for(AuthPropertyType auth: authList)
      {
         this.authPropsMap.put(auth.getKey(), auth.getValue());
      }
      
      this.keyStoreURL = this.authPropsMap.get(KEYSTORE_URL);
      this.keyStorePass = this.authPropsMap.get(KEYSTORE_PASS);
      

      this.signingAlias = this.authPropsMap.get(SIGNING_KEY_ALIAS);
      
      String keypass = this.authPropsMap.get(SIGNING_KEY_PASS);
      if(keypass == null || keypass.length() == 0)
         throw new RuntimeException("Signing Key Pass is null");
      this.signingKeyPass = keypass.toCharArray(); 
   }

   /**
    * @see TrustKeyManager#setValidatingAlias(List)
    */
   public void setValidatingAlias(List<KeyValueType> aliases)
   {
      for(KeyValueType alias: aliases)
      {
         domainAliasMap.put(alias.getKey(), alias.getValue());
      }
   }
   
   /**
    * @see TrustKeyManager#getEncryptionKey(String)
    */
   public SecretKey getEncryptionKey(String domain,String encryptionAlgorithm, int keyLength) throws Exception
   {
      SecretKey key = keys.get(domain);
      if(key == null)
      {
         key = EncryptionKeyUtil.getSecretKey(encryptionAlgorithm, keyLength);
         keys.put(domain, key);
      } 
      return key;
   }
   
   private void setUpKeyStore() throws GeneralSecurityException, IOException
   {
      //Keystore URL/Pass can be either by configuration or on the HTTPS connector
      if(this.keyStoreURL == null)
      {
         this.keyStoreURL = SecurityActions.getProperty("javax.net.ssl.keyStore", null);
      }
      if(this.keyStorePass == null)
      {
         this.keyStorePass = SecurityActions.getProperty("javax.net.ssl.keyStorePassword", null);
      }
      
      InputStream is = ValveUtil.getKeyStoreInputStream(this.keyStoreURL);
      ks = KeyStoreUtil.getKeyStore(is, keyStorePass.toCharArray()); 
   } 
}