/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.security.Key;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.util.Collections;

import javax.security.cert.X509Certificate;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.Marshaller;
import javax.xml.crypto.dsig.CanonicalizationMethod;
import javax.xml.crypto.dsig.Reference;
import javax.xml.crypto.dsig.SignedInfo;
import javax.xml.crypto.dsig.Transform;
import javax.xml.crypto.dsig.XMLSignature;
import javax.xml.crypto.dsig.XMLSignatureFactory;
import javax.xml.crypto.dsig.dom.DOMSignContext;
import javax.xml.crypto.dsig.dom.DOMValidateContext;
import javax.xml.crypto.dsig.keyinfo.KeyInfo;
import javax.xml.crypto.dsig.keyinfo.KeyInfoFactory;
import javax.xml.crypto.dsig.keyinfo.KeyValue;
import javax.xml.crypto.dsig.spec.C14NMethodParameterSpec;
import javax.xml.crypto.dsig.spec.TransformParameterSpec;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
 
import org.jboss.identity.federation.api.saml.v2.request.SAML2Request;
import org.jboss.identity.federation.core.saml.v2.factories.JBossSAMLBaseFactory;
import org.jboss.identity.federation.saml.v2.protocol.AuthnRequestType;
import org.jboss.identity.federation.saml.v2.protocol.RequestAbstractType;
import org.jboss.identity.xmlsec.w3.xmldsig.ObjectFactory;
import org.jboss.identity.xmlsec.w3.xmldsig.SignatureType;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

/**
 * Utility for XML Signature
 * @author Anil.Saldhana@redhat.com
 * @since Dec 15, 2008
 */
public class XMLSignatureUtil
{
   private static String pkgName = "org.jboss.identity.federation.w3.xmldsig";
   private static String schemaLocation = "schema/saml/v2/xmldsig-core-schema.xsd"; 
   
   private static ObjectFactory objectFactory = new ObjectFactory();
   
   private static XMLSignatureFactory fac =  getXMLSignatureFactory(); 
   
   private static XMLSignatureFactory getXMLSignatureFactory()
   {
      XMLSignatureFactory xsf =   null;
      
      try
      {
         xsf = XMLSignatureFactory.getInstance("DOM"); 
      } 
      catch(Exception err)
      {
         //JDK5
         xsf = XMLSignatureFactory.getInstance("DOM",
               new org.jcp.xml.dsig.internal.dom.XMLDSigRI());
      }
      return xsf;
   }
   
   /**
    * Sign an AuthnRequestType
    * @param request
    * @param signingKey Private Key for signing
    * @param cert X509Certificate public key certificate (may be null)
    * @param digestMethod (Example: DigestMethod.SHA1)
    * @param signatureMethod (Example: SignatureMethod.DSA_SHA1)
    * @return
    * @throws Exception
    */
   public static Document sign(AuthnRequestType request, PrivateKey signingKey,
         X509Certificate certificate,
         String digestMethod, String signatureMethod) throws Exception
   {
      DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance(); 
      dbf.setNamespaceAware(true); 
      
      SAML2Request saml2Request = new SAML2Request();
      
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      saml2Request.marshall(request, baos);
    
      DocumentBuilder builder = dbf.newDocumentBuilder();  
      Document doc = builder.parse(new ByteArrayInputStream(baos.toByteArray()) );
      
      DOMSignContext dsc = new DOMSignContext(signingKey, doc.getDocumentElement());   

      String referenceURI = "#" + request.getID();
      
      Reference ref = fac.newReference
        ( referenceURI, fac.newDigestMethod(digestMethod, null),
          Collections.singletonList
            (fac.newTransform(Transform.ENVELOPED,
              (TransformParameterSpec) null)), null, null); 
      
      SignedInfo si =  fac.newSignedInfo
      (fac.newCanonicalizationMethod
        (CanonicalizationMethod.INCLUSIVE_WITH_COMMENTS,
          (C14NMethodParameterSpec) null),
        fac.newSignatureMethod(signatureMethod, null),
        Collections.singletonList(ref));  
      
      KeyInfo ki = null;
      if(certificate != null)
      {
         KeyInfoFactory kif = fac.getKeyInfoFactory(); 
         KeyValue kv = kif.newKeyValue(certificate.getPublicKey());
         ki = kif.newKeyInfo(Collections.singletonList(kv)); 
      } 

      XMLSignature signature = fac.newXMLSignature(si, ki); 

      signature.sign(dsc); 
      
      return doc;
   }
   
   /**
    * Sign an RequestType
    * @param request
    * @param keypair Key Pair 
    * @param digestMethod (Example: DigestMethod.SHA1)
    * @param signatureMethod (Example: SignatureMethod.DSA_SHA1)
    * @return
    * @throws Exception
    */
   public static Document sign(RequestAbstractType request, KeyPair keypair,
         String digestMethod, String signatureMethod) throws Exception
   {
      DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance(); 
      dbf.setNamespaceAware(true); 
      
      SAML2Request saml2Request = new SAML2Request();
      
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      saml2Request.marshall(request, baos);
    
      DocumentBuilder builder = dbf.newDocumentBuilder();  
      Document doc = builder.parse(new ByteArrayInputStream(baos.toByteArray()) );
      
      DOMSignContext dsc = new DOMSignContext(keypair.getPrivate(), doc.getDocumentElement());   

      String referenceURI = "#" + request.getID();
      
      Reference ref = fac.newReference
        ( referenceURI, fac.newDigestMethod(digestMethod, null),
          Collections.singletonList
            (fac.newTransform(Transform.ENVELOPED,
              (TransformParameterSpec) null)), null, null); 
      
      SignedInfo si =  fac.newSignedInfo
      (fac.newCanonicalizationMethod
        (CanonicalizationMethod.INCLUSIVE_WITH_COMMENTS,
          (C14NMethodParameterSpec) null),
        fac.newSignatureMethod(signatureMethod, null),
        Collections.singletonList(ref));  
      
      KeyInfoFactory kif = fac.getKeyInfoFactory(); 
      KeyValue kv = kif.newKeyValue(keypair.getPublic());
      KeyInfo ki = kif.newKeyInfo(Collections.singletonList(kv)); 

      XMLSignature signature = fac.newXMLSignature(si, ki); 

      signature.sign(dsc); 
      
      return doc;
   }
   
   /**
    * Validate a signed document with the given public key
    * @param signedDoc
    * @param publicKey
    * @return
    * @throws Exception
    */
   public static boolean validate(Document signedDoc, Key publicKey) throws Exception
   {
      NodeList nl = signedDoc.getElementsByTagNameNS(XMLSignature.XMLNS, "Signature");
      if (nl.getLength() == 0) 
      {
        throw new Exception("Cannot find Signature element");
      } 
      DOMValidateContext valContext = new DOMValidateContext(publicKey, nl.item(0)); 
 
      XMLSignature signature =  fac.unmarshalXMLSignature(valContext); 
      boolean coreValidity = signature.validate(valContext); 

      return coreValidity;
   }
 
   /**
    * Marshall a SignatureType to output stream
    * @param signature
    * @param os
    * @throws Exception
    */
   public static void marshall(SignatureType signature, OutputStream os) throws Exception
   {
      JAXBElement<SignatureType> jsig = objectFactory.createSignature(signature);
      Marshaller marshaller = JBossSAMLBaseFactory.getValidatingMarshaller(pkgName, schemaLocation);
      marshaller.marshal(jsig, os);
   }
 
   /**
    * Marshall the signed document to an output stream
    * @param signedDocument
    * @param os
    * @throws Exception
    */
   public static void marshall(Document signedDocument, OutputStream os) throws Exception
   {
      TransformerFactory tf = TransformerFactory.newInstance();
      Transformer trans = tf.newTransformer();
      trans.transform(new DOMSource(signedDocument), new StreamResult(os)); 
   }
}