/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.wstrust;

import javax.xml.ws.handler.MessageContext;

import org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken;
import org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityTokenResponse;

/**
 * <p>
 * The {@code WSTrustRequestHandler} interface defines the methods that will be responsible for handling the different
 * types of WS-Trust request messages.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public interface WSTrustRequestHandler
{

   /**
    * <p>
    * Initializes the concrete {@code WSTrustRequestHandler} instance.
    * </p>
    * 
    * @param configuration a reference to object that contains the STS configuration.
    */
   public void initialize(STSConfiguration configuration);

   /**
    * <p>
    * Generates a security token according to the information specified in the request message and returns the created
    * token in the response.
    * </p>
    * 
    * @param request the security token request message.
    * @param context the context of the token request message. The context provides information that may be relevant to
    *            the request processing, such as the subject of the caller.
    * @return a {@code RequestSecurityTokenResponse} containing the generated token.
    * @throws WSTrustException if an error occurs while handling the request message.
    */
   public RequestSecurityTokenResponse issue(RequestSecurityToken request, MessageContext context)
         throws WSTrustException;

   /**
    * <p>
    * Renews the security token as specified in the request message, returning the renewed token in the response.
    * </p>
    * 
    * @param request the request message that contains the token to be renewed.
    * @param context the context of the token request message. The context provides information that may be relevant to
    *            the request processing, such as the subject of the caller.
    * @return a {@code RequestSecurityTokenResponse} containing the renewed token.
    * @throws WSTrustException if an error occurs while handling the renewal process.
    */
   public RequestSecurityTokenResponse renew(RequestSecurityToken request, MessageContext context)
         throws WSTrustException;

   /**
    * <p>
    * Cancels the security token as specified in the request message.
    * </p>
    * 
    * @param request the request message that contains the token to be canceled.
    * @param context the context of the token request message. The context provides information that may be relevant to
    *            the request processing, such as the subject of the caller.
    * @return a {@code RequestSecurityTokenResponse} indicating whether the token has been canceled or not.
    * @throws WSTrustException if an error occurs while handling the cancellation process.
    */
   public RequestSecurityTokenResponse cancel(RequestSecurityToken request, MessageContext context)
         throws WSTrustException;

   /**
    * <p>
    * Validates the security token as specified in the request message.
    * </p>
    * 
    * @param request the request message that contains the token to be validated.
    * @param context the context of the token request message. The context provides information that may be relevant to
    *            the request processing, such as the subject of the caller.
    * @return a {@code RequestSecurityTokenResponse} containing the validation status or a new token.
    * @throws WSTrustException if an error occurs while handling the validation process.
    */
   public RequestSecurityTokenResponse validate(RequestSecurityToken request, MessageContext context)
         throws WSTrustException;
}
