/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.wstrust;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Locale;

import javax.xml.bind.JAXBElement;

import org.jboss.identity.federation.ws.addressing.EndpointReferenceType;
import org.jboss.identity.federation.ws.policy.AppliesTo;
import org.jboss.identity.federation.ws.trust.LifetimeType;
import org.jboss.identity.federation.ws.wss.utility.AttributedDateTime;

/**
 * <p>
 * Utility class that provides methods for parsing/creating WS-Trust elements.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class WSTrustUtil
{

   /**
    * <p>
    * Parses the contents of the {@code AppliesTo} element and returns the address the uniquely identify the service
    * provider.
    * </p>
    * 
    * @param appliesTo the {@code AppliesTo} instance to be parsed.
    * @return the address of the service provider.
    */
   public static String parseAppliesTo(AppliesTo appliesTo)
   {
      EndpointReferenceType reference = null;
      for (Object obj : appliesTo.getAny())
      {
         if (obj instanceof EndpointReferenceType)
            reference = (EndpointReferenceType) obj;
         else if (obj instanceof JAXBElement)
         {
            JAXBElement<?> element = (JAXBElement<?>) obj;
            if (element.getName().getLocalPart().equalsIgnoreCase("EndpointReference"))
               reference = (EndpointReferenceType) element.getValue();
         }

         if (reference != null && reference.getAddress() != null)
            return reference.getAddress().toString();
      }
      return null;
   }

   /**
    * <p>
    * Creates a {@code LifetimeType} instance that specifies a range of time that starts at the current GMT time and has
    * the specified duration in milliseconds.
    * </p>
    * 
    * @param tokenTimeout the token timeout value (in milliseconds).
    * @return the constructed {@code LifetimeType} instance.
    */
   public static LifetimeType createDefaultLifetime(long tokenTimeout)
   {
      long createdTime = getCurrentGMTTime();
      final SimpleDateFormat calendarFormatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'.'SSS'Z'", Locale
            .getDefault());
      Calendar calendar = new GregorianCalendar();
      calendarFormatter.setTimeZone(calendar.getTimeZone());

      // instantiate the "created" time.
      calendar.setTimeInMillis(createdTime);
      AttributedDateTime created = new AttributedDateTime();
      created.setValue(calendarFormatter.format(calendar.getTime()));

      // instantiate the "expires" time.
      calendar.setTimeInMillis(createdTime + tokenTimeout);
      AttributedDateTime expires = new AttributedDateTime();
      expires.setValue(calendarFormatter.format(calendar.getTime()));

      LifetimeType type = new LifetimeType();
      type.setCreated(created);
      type.setExpires(expires);
      return type;
   }

   /**
    * <p>
    * Obtains the current GMT time in milliseconds.
    * </p>
    * 
    * @return a long representing the current GMT time in milliseconds.
    */
   public static long getCurrentGMTTime()
   {
      Calendar cal = new GregorianCalendar();
      int offset = cal.get(Calendar.ZONE_OFFSET);
      if (cal.getTimeZone().inDaylightTime(cal.getTime()))
         offset += cal.getTimeZone().getDSTSavings();

      // return the UTC/GMT time.
      return cal.getTimeInMillis() - offset;
   }
}
