/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.wstrust;

import java.net.URI;
import java.security.Principal;
import java.security.PublicKey;

import org.jboss.identity.federation.api.wstrust.plugins.saml.SAMLUtil;
import org.jboss.identity.federation.core.wstrust.RequestSecurityToken;
import org.jboss.identity.federation.core.wstrust.RequestSecurityTokenResponse;
import org.jboss.identity.federation.ws.policy.AppliesTo;
import org.jboss.identity.federation.ws.trust.RequestedSecurityTokenType;

/**
 * <p>
 * Default implementation of the {@code WSTrustRequestHandler} interface. It creates the request context containing the
 * original WS-Trust request as well as any information that may be relevant to the token processing, and delegates the
 * actual token handling processing to the appropriate {@code SecurityTokenProvider}.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class StandardRequestHandler implements WSTrustRequestHandler
{

   private STSConfiguration configuration;

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#initialize(org.jboss.identity.federation.api.wstrust.STSConfiguration)
    */
   public void initialize(STSConfiguration configuration)
   {
      this.configuration = configuration;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#issue(org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken,
    *      java.security.Principal)
    */
   public RequestSecurityTokenResponse issue(RequestSecurityToken request, Principal callerPrincipal)
         throws WSTrustException
   {
      SecurityTokenProvider provider = null;

      // first try to obtain the security token provider using the applies-to contents.
      AppliesTo appliesTo = request.getAppliesTo();
      PublicKey providerPublicKey = null;
      if (appliesTo != null)
      {
         String serviceName = WSTrustUtil.parseAppliesTo(appliesTo);
         if (serviceName != null)
         {
            provider = this.configuration.getProviderForService(serviceName);
            request.setTokenType(URI.create(this.configuration.getTokenTypeForService(serviceName)));
            providerPublicKey = this.configuration.getServiceProviderPublicKey(serviceName);
         }
      }
      // if applies-to is not available or if no provider was found for the service, use the token type.
      if (provider == null && request.getTokenType() != null)
      {
         provider = this.configuration.getProviderForTokenType(request.getTokenType().toString());
      }
      else if (appliesTo == null && request.getTokenType() == null)
         throw new WSTrustException("Either AppliesTo or TokenType must be present in a security token request");

      if (provider != null)
      {
         // create the request context and delegate token generation to the provider.
         WSTrustRequestContext requestContext = new WSTrustRequestContext(request, callerPrincipal);
         requestContext.setTokenIssuer(this.configuration.getSTSName());
         if (request.getLifetime() == null && this.configuration.getIssuedTokenTimeout() != 0)
         {
            // if no lifetime has been specified, use the configured timeout value.
            request.setLifetime(WSTrustUtil.createDefaultLifetime(this.configuration.getIssuedTokenTimeout()));
         }
         requestContext.setSTSKeyPair(this.configuration.getSTSKeyPair());
         requestContext.setServiceProviderPublicKey(providerPublicKey);
         provider.issueToken(requestContext);

         if (requestContext.getSecurityToken() == null)
            throw new WSTrustException("Token issued by provider " + provider.getClass().getName() + " is null");

         // construct the ws-trust security token response.
         RequestedSecurityTokenType requestedSecurityToken = new RequestedSecurityTokenType();
         requestedSecurityToken.setAny(requestContext.getSecurityToken().getTokenValue());

         // TODO: create proof token and encrypt the token if needed

         RequestSecurityTokenResponse response = new RequestSecurityTokenResponse();
         if (request.getContext() != null)
            response.setContext(request.getContext());

         response.setTokenType(request.getTokenType());
         response.setLifetime(request.getLifetime());
         response.setAppliesTo(appliesTo);
         response.setRequestedSecurityToken(requestedSecurityToken);

         // set the attached and unattached references.
         if (requestContext.getAttachedReference() != null)
            response.setRequestedAttachedReference(requestContext.getAttachedReference());
         if (requestContext.getUnattachedReference() != null)
            response.setRequestedUnattachedReference(requestContext.getUnattachedReference());

         return response;
      }
      else
         throw new WSTrustException("Unable to find a token provider for the token request");
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#renew(org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken,
    *      java.security.Principal)
    */
   public RequestSecurityTokenResponse renew(RequestSecurityToken request, Principal callerPrincipal)
         throws WSTrustException
   {
      // TODO: implement renew logic.
      throw new UnsupportedOperationException();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#validate(org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken,
    *      java.security.Principal)
    */
   public RequestSecurityTokenResponse validate(RequestSecurityToken request, Principal callerPrincipal)
         throws WSTrustException
   {
      if(request.getTokenType() == null)
         request.setTokenType(URI.create(WSTrustConstants.STATUS_TYPE));
      
      // TODO: get the provider using the token from the request.
      SecurityTokenProvider provider = this.configuration.getProviderForTokenType(SAMLUtil.SAML2_TOKEN_TYPE);
      WSTrustRequestContext context = new WSTrustRequestContext(request, callerPrincipal);
      context.setSTSKeyPair(this.configuration.getSTSKeyPair());
      
      provider.validateToken(context);
      
      RequestSecurityTokenResponse response = new RequestSecurityTokenResponse();
      if (request.getContext() != null)
         response.setContext(request.getContext());
      response.setTokenType(request.getTokenType());
      response.setStatus(context.getStatus());
      
      return response;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.WSTrustRequestHandler#cancel(org.jboss.identity.federation.api.wstrust.protocol.RequestSecurityToken,
    *      java.security.Principal)
    */
   public RequestSecurityTokenResponse cancel(RequestSecurityToken request, Principal callerPrincipal)
         throws WSTrustException
   {
      // TODO: implement cancel logic.
      throw new UnsupportedOperationException();
   }

}
