/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.wstrust.plugins.saml;

import java.net.URI;
import java.security.KeyPair;
import java.security.Principal;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.xml.crypto.dsig.DigestMethod;
import javax.xml.crypto.dsig.SignatureMethod;
import javax.xml.namespace.QName;

import org.jboss.identity.federation.api.util.XMLSignatureUtil;
import org.jboss.identity.federation.api.wstrust.SecurityToken;
import org.jboss.identity.federation.api.wstrust.SecurityTokenProvider;
import org.jboss.identity.federation.api.wstrust.StandardSecurityToken;
import org.jboss.identity.federation.api.wstrust.WSTrustConstants;
import org.jboss.identity.federation.api.wstrust.WSTrustException;
import org.jboss.identity.federation.api.wstrust.WSTrustRequestContext;
import org.jboss.identity.federation.api.wstrust.WSTrustUtil;
import org.jboss.identity.federation.core.exceptions.ConfigurationException;
import org.jboss.identity.federation.core.saml.v2.factories.SAMLAssertionFactory;
import org.jboss.identity.federation.core.saml.v2.util.AssertionUtil;
import org.jboss.identity.federation.core.wstrust.Lifetime;
import org.jboss.identity.federation.saml.v2.assertion.AssertionType;
import org.jboss.identity.federation.saml.v2.assertion.AudienceRestrictionType;
import org.jboss.identity.federation.saml.v2.assertion.ConditionsType;
import org.jboss.identity.federation.saml.v2.assertion.NameIDType;
import org.jboss.identity.federation.saml.v2.assertion.SubjectConfirmationType;
import org.jboss.identity.federation.saml.v2.assertion.SubjectType;
import org.jboss.identity.federation.ws.policy.AppliesTo;
import org.jboss.identity.federation.ws.trust.RequestedReferenceType;
import org.jboss.identity.federation.ws.trust.StatusType;
import org.jboss.identity.federation.ws.trust.ValidateTargetType;
import org.jboss.identity.federation.ws.wss.secext.KeyIdentifierType;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * <p>
 * A {@code SecurityTokenProvider} implementation that handles WS-Trust SAML 2.0 token requests.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class SAML20TokenProvider implements SecurityTokenProvider
{

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.SecurityTokenProvider#cancelToken(org.jboss.identity.federation.api.wstrust.WSTrustRequestContext)
    */
   public void cancelToken(WSTrustRequestContext context) throws WSTrustException
   {
      // TODO: implement cancel logic.
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.SecurityTokenProvider#issueToken(org.jboss.identity.federation.api.wstrust.WSTrustRequestContext)
    */
   public void issueToken(WSTrustRequestContext context) throws WSTrustException
   {
      // generate an id for the new assertion.
      String assertionID = "ID-" + UUID.randomUUID().toString();

      // lifetime and audience restrictions.
      Lifetime lifetime = context.getRequestSecurityToken().getLifetime();
      AudienceRestrictionType restriction = null;
      AppliesTo appliesTo = context.getRequestSecurityToken().getAppliesTo();
      if (appliesTo != null)
         restriction = SAMLAssertionFactory.createAudienceRestriction(WSTrustUtil.parseAppliesTo(appliesTo));
      ConditionsType conditions = SAMLAssertionFactory.createConditions(lifetime.getCreated(), lifetime.getExpires(),
            restriction);

      // TODO: implement support for the other confirmation methods.
      String confirmationMethod = SAMLUtil.SAML2_BEARER_URI;
      SubjectConfirmationType subjectConfirmation = SAMLAssertionFactory.createSubjectConfirmation(null,
            confirmationMethod, null);

      // create a subject using the caller principal.
      Principal principal = context.getCallerPrincipal();
      String subjectName = principal == null ? "ANONYMOUS" : principal.getName();
      NameIDType nameID = SAMLAssertionFactory.createNameID(null, "urn:jboss:identity-federation", subjectName);
      SubjectType subject = SAMLAssertionFactory.createSubject(nameID, subjectConfirmation);

      // TODO: add SAML statements that corresponds to the claims provided by the requester.

      // create the SAML assertion.
      NameIDType issuerID = SAMLAssertionFactory.createNameID(null, null, context.getTokenIssuer());
      AssertionType assertion = SAMLAssertionFactory.createAssertion(assertionID, issuerID, lifetime.getCreated(),
            conditions, subject, null);

      // convert the constructed assertion to element.
      Document document = null;
      try
      {
         document = SAMLUtil.toDocument(assertion);
      }
      catch (Exception e)
      {
         throw new WSTrustException("Failed to marshall SAMLV2 assertion", e);
      }

      // sign the generated SAML assertion.
      KeyPair keyPair = context.getSTSKeyPair();
      if (keyPair != null)
      {
         URI signatureURI = context.getRequestSecurityToken().getSignatureAlgorithm();
         String signatureMethod = signatureURI != null ? signatureURI.toString() : SignatureMethod.RSA_SHA1;
         try
         {
            XMLSignatureUtil.sign(document, keyPair, DigestMethod.SHA1, signatureMethod, "#" + assertionID);
         }
         catch (Exception e)
         {
            throw new WSTrustException("Failed to sign SAMLV2 assertion", e);
         }
      }

      SecurityToken token = new StandardSecurityToken(context.getRequestSecurityToken().getTokenType().toString(),
            document.getDocumentElement(), assertionID);
      context.setSecurityToken(token);

      // set the SAML assertion attached reference.
      KeyIdentifierType keyIdentifier = WSTrustUtil.createKeyIdentifier(SAMLUtil.SAML2_VALUE_TYPE, "#" + assertionID);
      Map<QName, String> attributes = new HashMap<QName, String>();
      attributes.put(new QName(WSTrustConstants.WSSE11_NS, "TokenType"), SAMLUtil.SAML2_TOKEN_TYPE);
      RequestedReferenceType attachedReference = WSTrustUtil.createRequestedReference(keyIdentifier, attributes);
      context.setAttachedReference(attachedReference);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.SecurityTokenProvider#renewToken(org.jboss.identity.federation.api.wstrust.WSTrustRequestContext)
    */
   public void renewToken(WSTrustRequestContext context) throws WSTrustException
   {
      // TODO: implement renew logic.
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.identity.federation.api.wstrust.SecurityTokenProvider#validateToken(org.jboss.identity.federation.api.wstrust.WSTrustRequestContext)
    */
   public void validateToken(WSTrustRequestContext context) throws WSTrustException
   {
      // get the SAML assertion that must be validated.
      ValidateTargetType validateTarget = context.getRequestSecurityToken().getValidateTarget();
      if(validateTarget == null)
         throw new WSTrustException("Invalid validate message: missing required ValidateTarget");
      Element assertionElement = (Element) validateTarget.getAny();
      
      String code = WSTrustConstants.STATUS_CODE_VALID;
      String reason = "SAMLV2.0 Assertion successfuly validated";
      
      if(!this.isAssertion(assertionElement))
      {
         code = WSTrustConstants.STATUS_CODE_INVALID;
         reason = "Validation failure: supplied token is not a SAMLV2.0 Assertion";
      }
      else
      {
         AssertionType assertion = null;

         // validate the SAML assertion digital signature.
         KeyPair keyPair = context.getSTSKeyPair();
         try
         {
            assertion = SAMLUtil.fromDocument(assertionElement.getOwnerDocument());
            if(!XMLSignatureUtil.validate(SAMLUtil.toDocument(assertion), keyPair.getPublic()))
            {
               code = WSTrustConstants.STATUS_CODE_INVALID;
               reason = "Validation failure: digital signature is invalid";
            }
         }
         catch (Exception e)
         {
            code = WSTrustConstants.STATUS_CODE_INVALID;
            reason = "Validation failure: unable to verify digital signature: " + e.getMessage();
         }
         
         // if the signature is valid, check the lifetime.
         try
         {
            if(AssertionUtil.hasExpired(assertion))
            {
               code = WSTrustConstants.STATUS_CODE_INVALID;
               reason = "Validation failure: assertion expired or used before its lifetime period";
            }
         }
         catch(ConfigurationException ce)
         {
            code = WSTrustConstants.STATUS_CODE_INVALID;
            reason = "Validation failure: unable to verify assertion lifetime: " + ce.getMessage();
         }
      }
      
      // construct the status and set it on the request context.
      StatusType status = new StatusType();
      status.setCode(code);
      status.setReason(reason);
      context.setStatus(status);
   }

   /**
    * <p>
    * Checks whether the specified element is a SAMLV2.0 assertion or not.
    * </p>
    *  
    * @param element the {@code Element} being verified.
    * @return {@code true} if the element is a SAMLV2.0 assertion; {@code false} otherwise.
    */
   private boolean isAssertion(Element element)
   {
      return element == null ? false : "Assertion".equals(element.getLocalName())
            && WSTrustConstants.SAML2_ASSERTION_NS.equals(element.getNamespaceURI());
   }
}
