/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.wstrust.plugins.saml;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMResult;

import org.jboss.identity.federation.api.wstrust.WSTrustConstants;
import org.jboss.identity.federation.saml.v2.assertion.AssertionType;
import org.jboss.identity.federation.saml.v2.assertion.ObjectFactory;
import org.w3c.dom.Document;

import com.sun.xml.bind.marshaller.NamespacePrefixMapper;

/**
 * <p>
 * This class contains utility methods and constants that are used by the SAML token providers.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class SAMLUtil
{

   public static final String SAML2_BEARER_URI = "urn:oasis:names:tc:SAML:2.0:cm:bearer";

   public static final String SAML2_TOKEN_TYPE = "http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.1#SAMLV2.0";

   public static final String SAML2_VALUE_TYPE = "http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.1#SAMLID";

   /**
    * <p>
    * Utility method that marshals the specified {@code AssertionType} object into a {@code Document} instance.
    * </p>
    * 
    * @param assertion an {@code AssertionType} object representing the SAML assertion to be marshaled.
    * @return a reference to the {@code Document} that contains the marshaled SAML assertion.
    * @throws Exception if an error occurs while marshaling the assertion.
    */
   public static Document toDocument(AssertionType assertion) throws Exception
   {
      Document document = null;
      document = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
      DOMResult result = new DOMResult(document);
      JAXBContext jaxbContext = JAXBContext.newInstance("org.jboss.identity.federation.saml.v2.assertion");
      Marshaller marshaller = jaxbContext.createMarshaller();
      marshaller.setProperty("com.sun.xml.bind.namespacePrefixMapper", new DefaultPrefixMapper());
      marshaller.marshal(new ObjectFactory().createAssertion(assertion), result);

      return document;
   }

   /**
    * <p>
    * Utility method that unmarshals the specified {@code Document} into an {@code AssertionType} instance.
    * </p>
    * 
    * @param document the {@code Document} that contains a marshaled SAMLV2.0 Assertion.
    * @return a reference to the unmarshaled {@code AssertionType} instance.
    * @throws JAXBException if an error occurs while unmarshaling the document.
    */
   public static AssertionType fromDocument(Document document) throws JAXBException
   {
      JAXBContext jaxbContext = JAXBContext.newInstance("org.jboss.identity.federation.saml.v2.assertion");
      Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
      Object object = unmarshaller.unmarshal(document);
      if (object instanceof AssertionType)
         return (AssertionType) object;
      else if (object instanceof JAXBElement)
      {
         JAXBElement<?> element = (JAXBElement<?>) object;
         if (element.getDeclaredType().equals(AssertionType.class))
            return (AssertionType) element.getValue();
      }
      throw new IllegalArgumentException("Supplied document does not contain a SAMLV2.0 Assertion");
   }

   /**
    * <p>
    * A {@code NamespacePrefixMapper} implementation that maps the most used namespaces to commonly used prefixes.
    * </p>
    * 
    * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
    */
   static class DefaultPrefixMapper extends NamespacePrefixMapper
   {
      @Override
      public String getPreferredPrefix(String namespaceURI, String suggestion, boolean requirePrefix)
      {
         if (WSTrustConstants.WSA_NS.equals(namespaceURI))
            return "wsa";
         else if (WSTrustConstants.WSU_NS.equals(namespaceURI))
            return "wsu";
         else if (WSTrustConstants.WSSE_NS.equals(namespaceURI))
            return "wsse";
         else if (WSTrustConstants.WSSE11_NS.equals(namespaceURI))
            return "wsse11";
         else if (WSTrustConstants.XENC_NS.equals(namespaceURI))
            return "xenc";
         else if (WSTrustConstants.DSIG_NS.equals(namespaceURI))
            return "ds";
         else if (WSTrustConstants.SAML2_ASSERTION_NS.equals(namespaceURI))
            return "saml2";
         else
            return null;
      }
   }
}
