/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.saml.v2.sig;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyPair;

import javax.xml.bind.JAXBException;
import javax.xml.crypto.MarshalException;
import javax.xml.crypto.dsig.DigestMethod;
import javax.xml.crypto.dsig.SignatureMethod;
import javax.xml.crypto.dsig.XMLSignatureException;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.xpath.XPathException;

import org.jboss.identity.federation.api.saml.v2.request.SAML2Request;
import org.jboss.identity.federation.api.saml.v2.response.SAML2Response;
import org.jboss.identity.federation.api.util.XMLSignatureUtil;
import org.jboss.identity.federation.core.saml.v2.constants.JBossSAMLURIConstants;
import org.jboss.identity.federation.core.saml.v2.util.DocumentUtil;
import org.jboss.identity.federation.saml.v2.protocol.RequestAbstractType;
import org.jboss.identity.federation.saml.v2.protocol.ResponseType;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

/**
 * Class that deals with SAML2 Signature
 * @author Anil.Saldhana@redhat.com
 * @since May 26, 2009
 */
public class SAML2Signature
{
   private String signatureMethod = SignatureMethod.RSA_SHA1;
   private String digestMethod = DigestMethod.SHA1;

   public String getSignatureMethod()
   {
      return signatureMethod;
   }

   public void setSignatureMethod(String signatureMethod)
   {
      this.signatureMethod = signatureMethod;
   }

   public String getDigestMethod()
   {
      return digestMethod;
   }

   public void setDigestMethod(String digestMethod)
   {
      this.digestMethod = digestMethod;
   }
   
   /**
    * Sign an RequestType at the root
    * @param request
    * @param keypair Key Pair 
    * @param digestMethod (Example: DigestMethod.SHA1)
    * @param signatureMethod (Example: SignatureMethod.DSA_SHA1)
    * @return 
    * @throws ParserConfigurationException 
    * @throws JAXBException 
    * @throws IOException 
    * @throws SAXException 
    * @throws XMLSignatureException 
    * @throws MarshalException 
    * @throws GeneralSecurityException 
    */
   public Document sign(RequestAbstractType request, KeyPair keypair) throws SAXException, IOException, JAXBException, ParserConfigurationException, GeneralSecurityException, MarshalException, XMLSignatureException  
   {
      SAML2Request saml2Request = new SAML2Request();
      Document doc = saml2Request.convert(request);
      doc.normalize();
      
      String referenceURI = "#" + request.getID();
       
      return XMLSignatureUtil.sign(doc, 
            keypair, 
            digestMethod, signatureMethod, 
            referenceURI);
   }
   
   /**
    * Sign an ResponseType at the root
    * @param response
    * @param keypair Key Pair 
    * @param digestMethod (Example: DigestMethod.SHA1)
    * @param signatureMethod (Example: SignatureMethod.DSA_SHA1)
    * @return 
    * @throws ParserConfigurationException 
    * @throws JAXBException 
    * @throws XMLSignatureException 
    * @throws MarshalException 
    * @throws GeneralSecurityException 
    */
   public Document sign(ResponseType response,KeyPair keypair) throws JAXBException, ParserConfigurationException, GeneralSecurityException, MarshalException, XMLSignatureException  
   {
      SAML2Response saml2Request = new SAML2Response();
      Document doc = saml2Request.convert(response);
      doc.normalize();
      
      String referenceURI = "#" + response.getID();
      
      return XMLSignatureUtil.sign(doc, 
            keypair, 
            digestMethod, signatureMethod, 
            referenceURI);
   }
   
   /**
    * Sign an assertion whose id value is provided in the response type
    * @param response
    * @param idValueOfAssertion
    * @param keypair
    * @param referenceURI
    * @return 
    * @throws ParserConfigurationException 
    * @throws JAXBException 
    * @throws TransformerException 
    * @throws TransformerFactoryConfigurationError 
    * @throws XPathException 
    * @throws XMLSignatureException 
    * @throws MarshalException 
    * @throws GeneralSecurityException 
    */
   public Document sign(ResponseType response,
         String idValueOfAssertion,
         KeyPair keypair, 
         String referenceURI) throws JAXBException, ParserConfigurationException, XPathException, TransformerFactoryConfigurationError, TransformerException, GeneralSecurityException, MarshalException, XMLSignatureException
   {
      SAML2Response saml2Request = new SAML2Response();
      Document doc = saml2Request.convert(response);
       
      
      Node assertionNode = DocumentUtil.getNodeWithAttribute(doc, 
            JBossSAMLURIConstants.ASSERTION_NSURI.get(), 
            "Assertion",
            "ID", 
            idValueOfAssertion);
      
      return XMLSignatureUtil.sign(doc, assertionNode, 
            keypair, 
            digestMethod, signatureMethod, 
            referenceURI);
   }
}