/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.spi;

import java.util.Collection;

import org.jboss.deployers.spi.management.ManagementView;
import org.jboss.deployers.spi.management.deploy.DeploymentManager;

/**
 * The entry point service for accessing/administering server profiles
 * 
 * @author Scott.Stark@jboss.org
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public interface ProfileService
{
   
   /**
    * Get the unique domains for which profiles exist.
    * 
    * @return array of profile domains
    */
   String[] getDomains();

   /**
    * Get the keys for all registered profiles.
    * 
    * @return keys for all registered profiles.
    */
   Collection<ProfileKey> getProfileKeys();

   /**
    * Obtain the registered profile for the key.
    * 
    * @param key - the key for the profile
    * @return the matching profile.
    * @throws NoSuchProfileException
    */
   Profile getProfile(ProfileKey key) throws NoSuchProfileException;

   /**
    * Get the keys for all active profiles.
    * 
    * @return keys for all active profiles.
    */
   Collection<ProfileKey> getActiveProfileKeys();
   
   /**
    * Obtain a activated profile for the key.
    * 
    * @param key - the key for the profile
    * @return the active profile
    * @throws NoSuchProfileException thrown if there is no
    *    active profile 
    */
   Profile getActiveProfile(ProfileKey key) throws NoSuchProfileException;
   
   /**
    * Obtain the ManagementView plugin
    * @return the ManagementView plugin if supported
    */
   ManagementView getViewManager();

   /**
    * Obtain the DeploymentManager plugin
    * @return the DeploymentManager plugin if supported
    */
   DeploymentManager getDeploymentManager();
   
   /**
    * Register a Profile.
    * 
    * @param profile the profile.
    * @throws Exception
    */
   void registerProfile(Profile profile) throws Exception;
   
   /**
    * Activate a registered profile.
    * 
    * @param key the profile key.
    * @throws NoSuchProfileException if the profile is not registered. 
    * @throws Exception
    */
   void activateProfile(ProfileKey key) throws NoSuchProfileException, Exception;
   
   /**
    * Deactivate the profile.
    * 
    * @param key the profile key.
    * @throws NoSuchProfileException if the profile is not active.
    */
   void deactivateProfile(ProfileKey key) throws NoSuchProfileException;

   /**
    * Unregister a profile.
    * 
    * @param key the profile key.
    * @throws NoSuchProfileException if the profile is not registered.
    */
   void unregisterProfile(ProfileKey key) throws NoSuchProfileException;

   /**
    * Validate a registered profile.
    * 
    * @param key the profile key
    * @throws Exception if the profile is not installed correctly
    */
   void validateProfile(ProfileKey key) throws Exception;
   
}
