/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.scanning.spi.jpa;

import java.lang.annotation.Annotation;
import java.net.URL;
import java.util.Set;

import org.jboss.scanning.spi.Resource;

/**
 * JPA environment.
 *
 * @author <a href="mailto:emmanuel@hibernate.org ">Emmanuel Bernard</a>
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public interface JPAEnvironment
{
   /**
    * Return all packages in the jar matching one of these annotations
    * if annotationsToLookFor is empty, return all packages.
    *
    * @param jarToScan jar to scan
    * @param annotationsToLookFor matching annotations
    * @return matching packages
    */
   Set<Package> getPackagesInJar(URL jarToScan, Class<? extends Annotation>... annotationsToLookFor);

   /**
    * Return all classes in the jar matching one of these annotations
    * if annotationsToLookFor is empty, return all classes.
    *
    * @param jarToScan jar to scan
    * @param annotationsToLookFor matching annotations
    * @return matching classes
    */
   Set<Class<?>> getClassesInJar(URL jarToScan, Class<? extends Annotation>... annotationsToLookFor);

   /**
    * Return all files in the jar matching one of these file names
    * if filePatterns is empty, return all files
    * e.g. *.hbm.xml, META-INF/orm.xml
    *
    * @param jarToScan jar to scan
    * @param filePatterns file patterns
    * @return resources
    */
   Set<Resource> getFilesInJar(URL jarToScan, String... filePatterns);

   /**
    * Return all files in the classpath (ie PU visibility)
    * matching one of these file names
    * if filePatterns is empty, return all files
    * the use case is really exact file name.
    *
    * @param jarToScan jar to scan
    * @param filePatterns file patterns
    * @return resources
    */
   Set<Resource> getFilesInClasspath(URL jarToScan, String... filePatterns);

   /**
    * Return the unqualified JAR name ie customer-model.jar or store.war.
    *
    * @param url the url
    * @return the unqualified name
    */
   String getUnqualifiedJarName(URL url);
}
