/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2011, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.dmr;

import java.io.IOException;
import java.io.InputStream;

import java.util.LinkedList;
import java.util.Vector;
import java.util.Stack;

public abstract class JSONParser
{
	protected final static int OPEN_BRACE = 256;
	protected final static int CLOSE_BRACE = 257;
	protected final static int OPEN_BRACKET = 258;
	protected final static int CLOSE_BRACKET = 259;
	protected final static int COLON = 260;
	protected final static int COMMA = 261;
	protected final static int BIG = 262;
	protected final static int INTEGER = 263;
	protected final static int DECIMAL = 264;
	protected final static int UNDEFINED = 265;
	protected final static int TRUE = 266;
	protected final static int FALSE = 267;
	protected final static int INT_VAL = 268;
	protected final static int INT_HEX_VAL = 269;
	protected final static int LONG_VAL = 270;
	protected final static int LONG_HEX_VAL = 271;
	protected final static int DOUBLE_SPECIAL_VAL = 272;
	protected final static int DEC_VAL = 273;
	protected final static int STR_VAL = 274;

	protected final static int INITIAL = 0;

	// an internal class for lazy initiation
	private final static class cc_lexer
	{
		private static char[] accept = ("\000\025\024\025\005\025\017\017\006\025\025\003\004\025\025\025\025\025\025\001\002\026\000\023\000\000\000\000\015\000\000\000\000\000\000\000\000\000\022\020\000\021\011\000\000\000\000\000\000\016\000\000\000\000\014\007\000\022\000\000\010\000\000\000\000\000\012\013").toCharArray ();
		private static char[] ecs = ("\000\000\000\000\000\000\000\000\000\001\002\000\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000\003\000\000\000\000\000\000\000\000\000\004\005\006\000\007\010\010\010\010\010\010\010\010\010\011\000\000\000\000\000\000\012\012\012\012\013\012\000\000\014\000\000\015\000\016\000\000\000\000\000\000\000\000\000\000\000\000\017\020\021\000\000\000\022\023\024\025\026\027\030\000\031\000\000\032\033\034\000\000\000\035\036\037\040\000\000\041\042\000\043\000\044\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\045").toCharArray ();
		private static char[] base = ("\047\000\003\000\000\030\000\001\000\007\001\000\000\014\067\002\062\057\063\000\000\000\000\000\000\000\000\002\000\007\072\104\073\100\073\067\075\070\000\114\101\000\000\102\076\107\104\111\020\000\104\106\114\113\000\000\007\000\113\123\000\120\110\116\114\110\000\000\153\000\155").toCharArray ();
		private static char[] next = ("\026\000\026\027\002\002\033\007\007\046\046\060\000\034\047\047\030\047\047\037\042\070\060\071\071\047\047\047\047\047\047\006\007\035\000\036\031\040\032\001\002\002\003\004\005\001\006\007\010\001\001\011\001\012\013\001\014\001\015\001\016\001\017\001\020\001\001\021\001\001\022\001\001\001\023\024\025\041\043\044\045\050\051\052\053\054\055\056\057\061\062\063\064\065\066\067\072\073\074\075\076\077\100\101\102\103\051\026\026\000\000\000\000\000\000\000\000\000\000\000\000").toCharArray ();
		private static char[] check = ("\105\106\105\003\002\002\006\006\006\033\033\046\070\006\035\035\003\035\035\012\017\060\046\060\060\035\035\035\035\035\035\005\005\006\007\011\005\015\005\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\016\020\021\022\036\037\040\041\042\043\044\045\047\050\053\054\055\056\057\062\063\064\065\072\073\075\076\077\100\101\104\104\106\106\106\106\106\106\106\106\106\106\106\106").toCharArray ();
		private static char[] defaults = ("\106\106\001\104\106\001\001\006\106\001\001\106\106\001\001\001\001\001\001\106\106\106\003\106\105\011\012\001\106\001\001\001\001\001\001\001\001\001\033\035\001\106\106\001\001\001\001\001\001\106\001\001\001\001\106\106\060\070\001\001\106\001\001\001\001\001\106\106\106\106\106").toCharArray ();
		private static char[] meta = ("\000\000\001\002\000\000\000\000\000\000\000\000\000\000\000\000\002\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\003").toCharArray ();
	}

	// an internal class for lazy initiation
	private final static class cc_parser
	{
		private static char[] rule = ("\000\001\001\001\001\001\001\001\001\002\002\003\002\002\003\001\001\002\003\004\005").toCharArray ();
		private static char[] ecs = ("\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\003\004\005\006\007\000\000\000\010\011\012\013\014\000\015\000\016\017").toCharArray ();
		private static char[] base = ("\010\124\011\006\030\031\041\042\052\053\063\001\004\000\142\064\002\074\124\075\025\105\000\000\106\145\116\035\115\000\141\155\164\176\170\176\176\176\176\176\176\176\176\176\176\176\176\176\176\176\176\176\171\176\176\172\176\176\176\176\176\173\176\176").toCharArray ();
		private static char[] next = ("\000\uffff\000\032\ufffe\023\ufff0\024\027\ufff0\001\ufff0\002\ufff0\021\033\003\004\005\006\007\010\011\012\ufffb\ufffa\030\ufffb\ufffa\ufffb\ufffa\ufffb\ufffa\ufffc\ufff9\035\ufffc\ufff9\ufffc\ufff9\ufffc\ufff9\ufff8\ufffd\000\ufff8\ufffd\ufff8\ufffd\ufff8\ufffd\ufff1\ufff4\000\ufff1\ufff4\ufff1\ufff4\ufff1\ufff4\ufff7\ufff6\000\ufff7\ufff6\ufff7\ufff6\ufff7\ufff6\ufff3\ufff5\000\ufff3\ufff5\ufff3\ufff5\ufff3\ufff5\ufff2\000\uffed\ufff2\000\ufff2\uffed\ufff2\000\017\000\uffef\000\uffef\000\000\000\000\000\000\000\020\uffec\025\000\000\uffec\026\uffee\000\uffee\000\000\000\000\000\000\000\000\013\014\015\016\000\022\031\034\036\000\000\000\000\000").toCharArray ();
		private static char[] check = ("\015\013\020\026\014\015\003\015\020\003\000\003\000\003\002\026\000\000\000\000\000\000\000\000\004\005\024\004\005\004\005\004\005\006\007\033\006\007\006\007\006\007\010\011\037\010\011\010\011\010\011\012\017\037\012\017\012\017\012\017\021\023\037\021\023\021\023\021\023\025\030\037\025\030\025\030\025\030\032\037\034\032\037\032\034\032\001\001\001\022\037\022\001\001\001\001\001\001\001\001\036\016\037\016\036\016\031\037\031\037\037\037\037\037\037\037\037\040\040\040\040\042\042\064\067\075\077\077\077\077\077").toCharArray ();
		private static char[] defaults = ("\037\000\000\037\003\003\003\003\003\003\003\037\013\013\015\003\013\003\015\003\000\003\001\000\003\015\003\020\016\000\016\037").toCharArray ();
		private static char[] gotoDefault = ("\077\077\040\077\077\077\077\077\077\077\077\077\077\077\077\077\077\077\077\077\042\077\077\042\077\077\077\077\077\042\077\077").toCharArray ();
		private static char[] lhs = ("\000\020\021\022\022\022\022\022\022\022\022\022\022\022\022\022\022\023\023\024\024").toCharArray ();
	}

	private final static class YYParserState	// internal tracking tool
	{
		int token;			// the current token type
		Object value;		// the current value associated with token
		int state;			// the current scan state

		YYParserState ()	// EOF token construction
		{
			this (0, null, 0);
		}
		YYParserState (int token)
		{
			this (token, null, 0);
		}
		YYParserState (int token, Object value)
		{
			this (token, value, 0);
		}
		YYParserState (int token, Object value, int state)
		{
			this.token = token;
			this.value = value;
			this.state = state;
		}
	}

	// lookahead stack for the parser
	private final LinkedList _yyLookaheadStack = new LinkedList ();
	// state stack for the parser
	private final Vector _yyStateStack = new Vector (512, 512);
	// flag that indicates error
	private boolean _yyInError;
	// internal track of the argument start
	private int _yyArgStart;
	// for passing value from lexer to parser
	private Object _yyValue;

	private InputStream _yyIs = System.in;
	private byte[] _yyBuffer;
	private int _yyBufferSize = 4096;
	private int _yyMatchStart;
	private int _yyBufferEnd;

	private int _yyBaseState;

	private int _yyTextStart;
	private int _yyLength;

	private Stack _yyLexerStack;
	private Stack _yyInputStack;


	/**
	 * Set the current input.
	 *
	 * @param	is
	 *			the new input.
	 */
	public void setInput (InputStream is)
	{
		_yyIs = is;
	}

	/**
	 * Obtain the current input.
	 *
	 * @return	the current input
	 */
	public InputStream getInput ()
	{
		return _yyIs;
	}

	/**
	 * Switch the current input to the new input.  The old input and already
	 * buffered characters are pushed onto the stack.
	 *
	 * @param	is
	 * 			the new input
	 */
	public void yyPushInput (InputStream is)
	{
		int len = _yyBufferEnd - _yyMatchStart;
		byte[] leftOver = new byte[len];
		System.arraycopy (_yyBuffer, _yyMatchStart, leftOver, 0, len);

		Object[] states = new Object[4];
		states[0] = _yyIs;
		states[1] = leftOver;

		if (_yyInputStack == null)
			_yyInputStack = new Stack ();
		_yyInputStack.push (states);

		_yyIs = is;
		_yyMatchStart = 0;
		_yyBufferEnd = 0;
	}

	/**
	 * Switch the current input to the old input on stack.  The currently
	 * buffered characters are inserted infront of the old buffered characters.
	 */
	public void yyPopInput ()
	{
		Object[] states = (Object[])_yyInputStack.pop ();
		_yyIs = (InputStream)states[0];
		byte[] leftOver = (byte[])states[1];

		int curLen = _yyBufferEnd - _yyMatchStart;

		if ((leftOver.length + curLen) > _yyBuffer.length)
		{
			byte[] newBuffer = new byte[leftOver.length + curLen];
			System.arraycopy (_yyBuffer, _yyMatchStart, newBuffer, 0, curLen);
			System.arraycopy (leftOver, 0, newBuffer, curLen, leftOver.length);
			_yyBuffer = newBuffer;
			_yyMatchStart = 0;
			_yyBufferEnd = leftOver.length + curLen;
		}
		else
		{
			int start = _yyMatchStart;
			int end = _yyBufferEnd;
			byte[] buffer = _yyBuffer;

			for (int i = 0; start < end; ++i, ++start)
				buffer[i] = buffer[start];
			System.arraycopy (leftOver, 0, buffer, curLen, leftOver.length);
			_yyMatchStart = 0;
			_yyBufferEnd = leftOver.length + curLen;
		}
	}

	/**
	 * Obtain the number of input objects on the stack.
	 *
	 * @return	the number of input objects on the stack.
	 */
	public int yyInputStackSize ()
	{
		return _yyInputStack == null ? 0 : _yyInputStack.size ();
	}


	/**
	 * Get the current token text.
	 * <p>
	 * Avoid calling this function unless it is absolutely necessary since it creates
	 * a copy of the token string.  The string length can be found by reading _yyLength
	 * or calling yyLength () function.
	 *
	 * @return	the current text token.
	 */
	public String yyText ()
	{
		if (_yyMatchStart == _yyTextStart)		// this is the case when we have EOF
			return null;
		return new String (_yyBuffer, _yyTextStart, _yyMatchStart - _yyTextStart);
	}

	/**
	 * Get the current text token's length.  Actions specified in the CookCC file
	 * can directly access the variable _yyLength.
	 *
	 * @return	the string token length
	 */
	public int yyLength ()
	{
		return _yyLength;
	}

	/**
	 * Print the current string token to the standard output.
	 */
	public void echo ()
	{
		System.out.print (yyText ());
	}

	/**
	 * Put all but n characters back to the input stream.  Be aware that calling
	 * yyLess (0) is allowed, but be sure to change the state some how to avoid
	 * an endless loop.
	 *
	 * @param	n
	 * 			The number of characters.
	 */
	protected void yyLess (int n)
	{
		if (n < 0)
			throw new IllegalArgumentException ("yyLess function requires a non-zero value.");
		if (n > (_yyMatchStart - _yyTextStart))
			throw new IndexOutOfBoundsException ("yyLess function called with a too large index value " + n + ".");
		_yyMatchStart = _yyTextStart + n;
	}

	/**
	 * Set the lexer's current state.
	 *
	 * @param	baseState
	 *			the base state index
	 */
	protected void begin (int baseState)
	{
		_yyBaseState = baseState;
	}

	/**
	 * Push the current state onto lexer state onto stack and
	 * begin the new state specified by the user.
	 *
	 * @param	newState
	 *			the new state.
	 */
	protected void yyPushLexerState (int newState)
	{
		if (_yyLexerStack == null)
			_yyLexerStack = new Stack ();
		_yyLexerStack.push (new Integer (_yyBaseState));
		begin (newState);
	}

	/**
	 * Restore the previous lexer state.
	 */
	protected void yyPopLexerState ()
	{
		begin (((Integer)_yyLexerStack.pop ()).intValue ());
	}


	// read more data from the input
	protected boolean yyRefreshBuffer () throws IOException
	{
		if (_yyBuffer == null)
			_yyBuffer = new byte[_yyBufferSize];
		if (_yyMatchStart > 0)
		{
			if (_yyBufferEnd > _yyMatchStart)
			{
				System.arraycopy (_yyBuffer, _yyMatchStart, _yyBuffer, 0, _yyBufferEnd - _yyMatchStart);
				_yyBufferEnd -= _yyMatchStart;
				_yyMatchStart = 0;
			}
			else
			{
				_yyMatchStart = 0;
				_yyBufferEnd = 0;
			}
		}
		else if (_yyBufferEnd == _yyBuffer.length)
		{
			byte[] newBuffer = new byte[_yyBuffer.length + _yyBuffer.length / 2];
			System.arraycopy (_yyBuffer, 0, newBuffer, 0, _yyBufferEnd);
			_yyBuffer = newBuffer;
		}

		int readSize = _yyIs.read (_yyBuffer, _yyBufferEnd, _yyBuffer.length - _yyBufferEnd);
		if (readSize > 0)
			_yyBufferEnd += readSize;
		else if (readSize < 0 && !yyWrap ())		// since we are at EOF, call yyWrap ().  If the return value of yyWrap is false, refresh buffer again
			return yyRefreshBuffer ();
		return readSize >= 0;
	}

	/**
	 * Reset the internal buffer.
	 */
	public void yyResetBuffer ()
	{
		_yyMatchStart = 0;
		_yyBufferEnd = 0;
	}

	/**
	 * Set the internal buffer size.  This action can only be performed
	 * when the buffer is empty.  Having a large buffer is useful to read
	 * a whole file in to increase the performance sometimes.
	 *
	 * @param	bufferSize
	 *			the new buffer size.
	 */
	public void setBufferSize (int bufferSize)
	{
		if (_yyBufferEnd > _yyMatchStart)
			throw new IllegalArgumentException ("Cannot change lexer buffer size at this moment.");
		_yyBufferSize = bufferSize;
		_yyMatchStart = 0;
		_yyBufferEnd = 0;
		if (_yyBuffer != null && bufferSize != _yyBuffer.length)
			_yyBuffer = new byte[bufferSize];
	}

	/**
	 * Call this function to start the scanning of the input.
	 *
	 * @return	a token or status value.
	 * @throws	IOException
	 *			in case of I/O error.
	 */
	protected int yyLex () throws IOException
	{

		char[] cc_ecs = cc_lexer.ecs;
		char[] cc_next = cc_lexer.next;
		char[] cc_check = cc_lexer.check;
		char[] cc_base = cc_lexer.base;
		char[] cc_default = cc_lexer.defaults;
		char[] cc_meta = cc_lexer.meta;
		char[] cc_accept = cc_lexer.accept;

		byte[] buffer = _yyBuffer;

		while (true)
		{
			// initiate variables necessary for lookup
			int cc_matchedState = _yyBaseState;

			int matchedLength = 0;

			int internalBufferEnd = _yyBufferEnd;
			int lookahead = _yyMatchStart;

			int cc_backupMatchedState = cc_matchedState;
			int cc_backupMatchedLength = 0;

			// the DFA lookup
			while (true)
			{
				// check buffer status
				if (lookahead < internalBufferEnd)
				{
					// now okay to process the character
					int cc_toState;
					int symbol = cc_ecs[buffer[lookahead] & 0xff];
					cc_toState = cc_matchedState;
					while (cc_check[symbol + cc_base[cc_toState]] != cc_toState)
					{
						cc_toState = cc_default[cc_toState];
						if (cc_toState >= 68)
							symbol = cc_meta[symbol];
					}
					cc_toState = cc_next[symbol + cc_base[cc_toState]];

					if (cc_toState == 0)
					{
						cc_matchedState = cc_backupMatchedState;
						matchedLength = cc_backupMatchedLength;
						break;
					}

					cc_matchedState = cc_toState;
					++lookahead;
					++matchedLength;

					if (cc_accept[cc_matchedState] > 0)
					{
						cc_backupMatchedState = cc_toState;
						cc_backupMatchedLength = matchedLength;
					}
				}
				else
				{
					int lookPos = lookahead - _yyMatchStart;
					boolean refresh = yyRefreshBuffer ();
					buffer = _yyBuffer;
					internalBufferEnd = _yyBufferEnd;
					lookahead = _yyMatchStart + lookPos;
					if (! refresh)
					{
						// <<EOF>>
						int cc_toState;
						int symbol = cc_ecs[256];
						cc_toState = cc_matchedState;
						while (cc_check[symbol + cc_base[cc_toState]] != cc_toState)
						{
							cc_toState = cc_default[cc_toState];
							if (cc_toState >= 68)
								symbol = cc_meta[symbol];
						}
						cc_toState = cc_next[symbol + cc_base[cc_toState]];

						if (cc_toState != 0)
							cc_matchedState = cc_toState;
						else
						{
							cc_matchedState = cc_backupMatchedState;
							matchedLength = cc_backupMatchedLength;
						}
						break;
					}
				}
			}

			_yyTextStart = _yyMatchStart;
			_yyMatchStart += matchedLength;
			_yyLength = matchedLength;


			switch (cc_accept[cc_matchedState])
			{
				case 1:	// \{
				{
					_yyValue = m_this.parseToken (); return OPEN_BRACE;
				}
				case 26: break;
				case 2:	// \}
				{
					_yyValue = m_this.parseToken (); return CLOSE_BRACE;
				}
				case 27: break;
				case 3:	// \[
				{
					_yyValue = m_this.parseToken (); return OPEN_BRACKET;
				}
				case 28: break;
				case 4:	// \]
				{
					_yyValue = m_this.parseToken (); return CLOSE_BRACKET;
				}
				case 29: break;
				case 5:	// ,
				{
					_yyValue = m_this.parseToken (); return COMMA;
				}
				case 30: break;
				case 6:	// :
				{
					_yyValue = m_this.parseToken (); return COLON;
				}
				case 31: break;
				case 7:	// true
				{
					_yyValue = m_this.parseToken (); return TRUE;
				}
				case 32: break;
				case 8:	// false
				{
					_yyValue = m_this.parseToken (); return FALSE;
				}
				case 33: break;
				case 9:	// big
				{
					_yyValue = m_this.parseToken (); return BIG;
				}
				case 34: break;
				case 10:	// decimal
				{
					_yyValue = m_this.parseToken (); return DECIMAL;
				}
				case 35: break;
				case 11:	// integer
				{
					_yyValue = m_this.parseToken (); return INTEGER;
				}
				case 36: break;
				case 12:	// null
				{
					_yyValue = m_this.parseToken (); return UNDEFINED;
				}
				case 37: break;
				case 13:	// [+-]?[0-9]+L
				{
					_yyValue = m_this.parsePlainValue (); return LONG_VAL;
				}
				case 38: break;
				case 14:	// [+-]?0x[0-9a-fA-F]+L
				{
					_yyValue = m_this.parsePlainValue (); return LONG_HEX_VAL;
				}
				case 39: break;
				case 15:	// [+-]?[0-9]+
				{
					_yyValue = m_this.parsePlainValue (); return INT_VAL;
				}
				case 40: break;
				case 16:	// [+-]?0x[0-9a-fA-F]+
				{
					_yyValue = m_this.parsePlainValue (); return INT_HEX_VAL;
				}
				case 41: break;
				case 17:	// [+-]?(NaN|Infinity)
				{
					_yyValue = m_this.parsePlainValue (); return DOUBLE_SPECIAL_VAL;
				}
				case 42: break;
				case 18:	// [+-]?([0-9]+\.[0-9]+([eE][+-]?[0-9]+)?)
				{
					_yyValue = m_this.parsePlainValue (); return DEC_VAL;
				}
				case 43: break;
				case 19:	// \"([^"\\]+|\\.)*\"
				{
					_yyValue = m_this.parseStringValue (); return STR_VAL;
				}
				case 44: break;
				case 20:	// [ \t\r\n]+
				{
					m_this.ignored ();
				}
				case 45: break;
				case 21:	// .
				{
					m_this.invalid ();
				}
				case 46: break;
				case 22:	// <<EOF>>
				{
					m_this.parseEOF (); return 0;
				}
				case 47: break;
				case 23:	// .|\n
				{
					echo ();			// default character action
				}
				case 48: break;
				case 24:	// <<EOF>>
				{
					return 0;			// default EOF action
				}
				case 49: break;
				default:
					throw new IOException ("Internal error in JSONParser lexer.");
			}

		}
	}


	/**
	 * Call this function to start parsing.
	 *
	 * @return	0 if everything is okay, or 1 if an error occurred.
	 * @throws	IOException
	 *			in case of error
	 */
	public int yyParse () throws IOException
	{
		char[] cc_ecs = cc_parser.ecs;
		char[] cc_next = cc_parser.next;
		char[] cc_check = cc_parser.check;
		char[] cc_base = cc_parser.base;
		char[] cc_default = cc_parser.defaults;
		char[] cc_gotoDefault = cc_parser.gotoDefault;
		char[] cc_rule = cc_parser.rule;
		char[] cc_lhs = cc_parser.lhs;

		LinkedList cc_lookaheadStack = _yyLookaheadStack;
		Vector cc_stateStack = _yyStateStack;

		if (cc_stateStack.size () == 0)
			cc_stateStack.add (new YYParserState ());

		int cc_toState = 0;

		for (;;)
		{
			YYParserState cc_lookahead;

			int cc_fromState;
			char cc_ch;

			//
			// check if there are any lookahead tokens on stack
			// if not, then call yyLex ()
			//
			if (cc_lookaheadStack.size () == 0)
			{
				_yyValue = null;
				int val = yyLex ();
				cc_lookahead = new YYParserState (val, _yyValue);
				cc_lookaheadStack.add (cc_lookahead);
			}
			else
				cc_lookahead = (YYParserState)cc_lookaheadStack.getLast ();

			cc_ch = cc_ecs[cc_lookahead.token];
			cc_fromState = ((YYParserState)cc_stateStack.get (cc_stateStack.size () - 1)).state;
			int cc_symbol = cc_ch;
			cc_toState = cc_fromState;
			while (cc_check[cc_symbol + cc_base[cc_toState]] != cc_toState)
			{
				cc_toState = cc_default[cc_toState];
				if (cc_toState >= 31)
					cc_symbol = 0;
			}
			cc_toState = (short)cc_next[cc_symbol + cc_base[cc_toState]];


			//
			// check the value of toState and determine what to do
			// with it
			//
			if (cc_toState > 0)
			{
				// shift
				cc_lookahead.state = cc_toState;
				cc_stateStack.add (cc_lookahead);
				cc_lookaheadStack.removeLast ();
				continue;
			}
			else if (cc_toState == 0)
			{
				// error
				if (_yyInError)
				{
					// first check if the error is at the lookahead
					if (cc_ch == 1)
					{
						// so we need to reduce the stack until a state with reduceable
						// action is found
						if (_yyStateStack.size () > 1)
							_yyStateStack.setSize (_yyStateStack.size () - 1);
						else
							return 1;	// can't do much we exit the parser
					}
					else
					{
						// this means that we need to dump the lookahead.
						if (cc_ch == 0)		// can't do much with EOF;
							return 1;
						cc_lookaheadStack.removeLast ();
					}
					continue;
				}
				else
				{
					if (yyParseError (cc_lookahead.token))
						return 1;
					_yyLookaheadStack.add (new YYParserState (1, _yyValue));
					_yyInError = true;
					continue;
				}
			}
			_yyInError = false;
			// now the reduce action
			int cc_ruleState = -cc_toState;

			_yyArgStart = cc_stateStack.size () - cc_rule[cc_ruleState] - 1;
			//
			// find the state that said need this non-terminal
			//
			cc_fromState = ((YYParserState)cc_stateStack.get (_yyArgStart)).state;

			//
			// find the state to goto after shifting the non-terminal
			// onto the stack.
			//
			if (cc_ruleState == 1)
				cc_toState = 0;			// reset the parser
			else
			{
				cc_toState = cc_fromState + 32;
				int cc_tmpCh = cc_lhs[cc_ruleState] - 16;
				while (cc_check[cc_tmpCh + cc_base[cc_toState]] != cc_toState)
					cc_toState = cc_gotoDefault[cc_toState - 32];
				cc_toState = cc_next[cc_tmpCh + cc_base[cc_toState]];
			}

			_yyValue = null;

			switch (cc_ruleState)
			{
				case 1:					// accept
					return 0;
				case 2:	// complete : node
				{
					return m_this.parse ((org.jboss.dmr.ModelNode)yyGetValue (1));
				}
				case 23: break;
				case 3:	// node : DEC_VAL
				{
					_yyValue = m_this.parseBigDecimal ((java.lang.String)yyGetValue (1));
				}
				case 24: break;
				case 4:	// node : INT_VAL
				{
					_yyValue = m_this.parseBigInteger ((java.lang.String)yyGetValue (1));
				}
				case 25: break;
				case 5:	// node : TRUE
				{
					_yyValue = m_this.parseBoolean ();
				}
				case 26: break;
				case 6:	// node : FALSE
				{
					_yyValue = m_this.parseBoolean ();
				}
				case 27: break;
				case 7:	// node : INT_HEX_VAL
				{
					_yyValue = m_this.parseIntHex ((java.lang.String)yyGetValue (1));
				}
				case 28: break;
				case 8:	// node : LONG_HEX_VAL
				{
					_yyValue = m_this.parseLongHex ((java.lang.String)yyGetValue (1));
				}
				case 29: break;
				case 9:	// node : OPEN_BRACKET CLOSE_BRACKET
				{
					_yyValue = m_this.parseEmptyList ();
				}
				case 30: break;
				case 10:	// node : list CLOSE_BRACKET
				{
					_yyValue = m_this.finishList ((org.jboss.dmr.ModelNode)yyGetValue (1));
				}
				case 31: break;
				case 11:	// node : list COMMA CLOSE_BRACKET
				{
					_yyValue = m_this.finishList ((org.jboss.dmr.ModelNode)yyGetValue (1));
				}
				case 32: break;
				case 12:	// node : OPEN_BRACE CLOSE_BRACE
				{
					_yyValue = m_this.parseEmptyObject ();
				}
				case 33: break;
				case 13:	// node : object CLOSE_BRACE
				{
					_yyValue = m_this.finishObject ((org.jboss.dmr.ModelNode)yyGetValue (1));
				}
				case 34: break;
				case 14:	// node : object COMMA CLOSE_BRACE
				{
					_yyValue = m_this.finishObject ((org.jboss.dmr.ModelNode)yyGetValue (1));
				}
				case 35: break;
				case 15:	// node : STR_VAL
				{
					_yyValue = m_this.parseString ((java.lang.String)yyGetValue (1));
				}
				case 36: break;
				case 16:	// node : UNDEFINED
				{
					_yyValue = m_this.parseUndefined ();
				}
				case 37: break;
				case 17:	// list : OPEN_BRACKET node
				{
					_yyValue = m_this.parseStartList ((org.jboss.dmr.ModelNode)yyGetValue (2));
				}
				case 38: break;
				case 18:	// list : list COMMA node
				{
					_yyValue = m_this.parseListItem ((org.jboss.dmr.ModelNode)yyGetValue (1), (org.jboss.dmr.ModelNode)yyGetValue (3));
				}
				case 39: break;
				case 19:	// object : OPEN_BRACE STR_VAL COLON node
				{
					_yyValue = m_this.parseStartObject ((java.lang.String)yyGetValue (2), (org.jboss.dmr.ModelNode)yyGetValue (4));
				}
				case 40: break;
				case 20:	// object : object COMMA STR_VAL COLON node
				{
					_yyValue = m_this.parseObjectItem ((org.jboss.dmr.ModelNode)yyGetValue (1), (java.lang.String)yyGetValue (3), (org.jboss.dmr.ModelNode)yyGetValue (5));
				}
				case 41: break;
				default:
					throw new IOException ("Internal error in JSONParser parser.");
			}

			YYParserState cc_reduced = new YYParserState (-cc_ruleState, _yyValue, cc_toState);
			_yyValue = null;
			cc_stateStack.setSize (_yyArgStart + 1);
			cc_stateStack.add (cc_reduced);
		}
	}

	/**
	 * This function is used by the error handling grammars to check the immediate
	 * lookahead token on the stack.
	 *
	 * @return	the top of lookahead stack.
	 */
	protected YYParserState yyPeekLookahead ()
	{
		return (YYParserState)_yyLookaheadStack.getLast ();
	}

	/**
	 * This function is used by the error handling grammars to pop an unwantted
	 * token from the lookahead stack.
	 */
	protected void yyPopLookahead ()
	{
		_yyLookaheadStack.removeLast ();
	}

	/**
	 * Clear the error flag.  If this flag is present and the parser again sees
	 * another error transition, it would immediately calls yyParseError, which
	 * would by default exit the parser.
	 * <p>
	 * This function is used in error recovery.
	 */
	protected void yyClearError ()
	{
		_yyInError = false;
	}

	/**
	 * This function reports error and return true if critical error occurred, or
	 * false if the error has been successfully recovered.  IOException is an optional
	 * choice of reporting error.
	 *
	 * @param	terminal
	 *			the terminal that caused the error.
	 * @return	true if irrecoverable error occurred.  Or simply throw an IOException.
	 *			false if the parsing can be continued to check for specific
	 *			error tokens.
	 * @throws	IOException
	 *			in case of error.
	 */
	protected boolean yyParseError (int terminal) throws IOException
	{
		return false;
	}

	/**
	 * Gets the object value associated with the symbol at the argument's position.
	 *
	 * @param	arg
	 *			the symbol position starting from 1.
	 * @return	the object value associated with symbol.
	 */
	protected Object yyGetValue (int arg)
	{
		return ((YYParserState)_yyStateStack.get (_yyArgStart + arg)).value;
	}

	/**
	 * Set the object value for the current non-terminal being reduced.
	 *
	 * @param	value
	 * 			the object value for the current non-terminal.
	 */
	protected void yySetValue (Object value)
	{
		_yyValue = value;
	}




	private final org.jboss.dmr.JSONParserImpl m_this = (org.jboss.dmr.JSONParserImpl)this;

	/**
	 * This function is used to change the initial state for the lexer.
	 *
	 * @param	state
	 *			the name of the state
	 */
	protected void begin (String state)
	{
		if ("INITIAL".equals (state))
		{
			begin (INITIAL);
			return;
		}
		throw new IllegalArgumentException ("Unknown lexer state: " + state);
	}

	/**
	 * Push the current state onto lexer state onto stack and
	 * begin the new state specified by the user.
	 *
	 * @param	state
	 *			the new state.
	 */
	protected void yyPushLexerState (String state)
	{
		if ("INITIAL".equals (state))
		{
			yyPushLexerState (INITIAL);
			return;
		}
		throw new IllegalArgumentException ("Unknown lexer state: " + state);
	}

	/**
	 * Check if there are more inputs.  This function is called when EOF is
	 * encountered.
	 *
	 * @return	true to indicate no more inputs.
	 * @throws	IOException
	 * 			in case of an IO error
	 */
	protected boolean yyWrap () throws IOException
	{
		return true;
	}


/*
 * lexer properties:
 * unicode = false
 * bol = false
 * backup = true
 * cases = 24
 * table = compressed
 * ecs = 38
 * states = 68
 * max symbol value = 256
 *
 * memory usage:
 * full table = 17476
 * ecs table = 2841
 * next = 121
 * check = 121
 * default = 71
 * meta = 38
 * compressed table = 608
 *
 * parser properties:
 * symbols = 21
 * max terminal = 274
 * used terminals = 16
 * non-terminals = 5
 * rules = 20
 * shift/reduce conflicts = 0
 * reduct/reduce conflicts = 0
 *
 * memory usage:
 * ecs table = 926
 * compressed table = 665
 */
}
