/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.ejb.jndi;

import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.log4j.Logger;

/**
 * @author ochaloup
 */
public abstract class InitialContextDirectory {

    public static final String CLASSLOADER_SCAN_PROPERTY_NAME
            = "jboss.ejb.client.properties.skip.classloader.scan";

    private static final Logger log = Logger.getLogger(InitialContextDirectory.class);
    protected InitialContext ctx;

    /**
     * Looking for a bean
     *
     * @param earName       name of EAR file where the bean is placed to, whether does not exist pass null
     * @param jarName       name of JAR file where the bean is placed to
     * @param beanClass     class to be looked for (there could be better to say Class<? extends T> but it does not work for EJB2 beans)
     * @param beanInterface interface of the class
     * @param beanType      type of bean that we are searching for
     * @param distinctName
     * @return returning remote stub interface
     */
    public abstract <T> T lookup(String earName, String jarName, Class<?> beanClass, Class<T> beanInterface,
                                 BeanType beanType, String distinctName) throws NamingException ;

    /**
     * Looking for stateful bean in ear.
     */
    public <T> T lookupStateful(String earName, String jarName, Class<? extends T> beanClass,
                                Class<T> beanInterface) throws NamingException {
        return lookup(earName, jarName, beanClass, beanInterface, BeanType.STATEFUL, null);
    }

    /**
     * Looking for stateful bean in ear, with distinct-name
     */
    public <T> T lookupStatefulWithDN(String earName, String jarName, Class<? extends T> beanClass,
                                      Class<T> beanInterface, String distinctName) throws NamingException {
        return lookup(earName, jarName, beanClass, beanInterface, BeanType.STATEFUL, distinctName);
    }

    /**
     * Looking for stateful bean in jar.
     */
    public <T> T lookupStateful(String jarName, Class<? extends T> beanClass, Class<T> beanInterface)
            throws NamingException {
        return lookupStateful(null, jarName, beanClass, beanInterface);
    }

    /**
     * Looking for stateful bean in jar, with distinct-name
     */
    public <T> T lookupStatefulWithDN(String jarName, Class<? extends T> beanClass, Class<T> beanInterface,
                                      String distinctName) throws NamingException {
        return lookupStatefulWithDN(null, jarName, beanClass, beanInterface, distinctName);
    }

    /**
     * Looking for stateless bean in ear.
     */
    public <T> T lookupStateless(String earName, String jarName, Class<? extends T> beanClass,
                                 Class<T> beanInterface) throws NamingException {
        return lookup(earName, jarName, beanClass, beanInterface, BeanType.STATELESS, null);
    }

    /**
     * Looking for stateless bean in ear, with distinct-name
     */
    public <T> T lookupStatelessWithDN(String earName, String jarName, Class<? extends T> beanClass,
                                       Class<T> beanInterface, String distinctName) throws NamingException {
        return lookup(earName, jarName, beanClass, beanInterface, BeanType.STATELESS, distinctName);
    }

    /**
     * Looking for stateless bean in jar.
     */
    public <T> T lookupStateless(String jarName, Class<? extends T> beanClass, Class<T> beanInterface)
            throws NamingException {
        return lookupStateless(null, jarName, beanClass, beanInterface);
    }

    /**
     * Looking for stateless bean in jar.
     */
    public <T> T lookupStatelessWithDN(String jarName, Class<? extends T> beanClass, Class<T> beanInterface,
                                       String distinctName) throws NamingException {
        return lookupStatelessWithDN(null, jarName, beanClass, beanInterface, distinctName);
    }


    /**
     * Looking for singleton class in ear.
     */
    public <T> T lookupSingleton(String earName, String jarName, Class<? extends T> beanClass,
                                 Class<T> beanInterface) throws NamingException {
        return lookup(earName, jarName, beanClass, beanInterface, BeanType.SINGLETON, null);
    }

    /**
     * Looking for singleton class in ear, with distinct-name
     */
    public <T> T lookupSingletonWithDN(String earName, String jarName, Class<? extends T> beanClass,
                                       Class<T> beanInterface, String distinctName) throws NamingException {
        return lookup(earName, jarName, beanClass, beanInterface, BeanType.SINGLETON, distinctName);
    }

    /**
     * Looking for singleton class in jar.
     */
    public <T> T lookupSingleton(String jarName, Class<? extends T> beanClass, Class<T> beanInterface)
            throws NamingException {
        return lookupSingleton(null, jarName, beanClass, beanInterface);
    }

    /**
     * Looking for singleton class in jar.
     */
    public <T> T lookupSingletonWithDN(String jarName, Class<? extends T> beanClass, Class<T> beanInterface,
                                       String distinctName) throws NamingException {
        return lookupSingletonWithDN(null, jarName, beanClass, beanInterface, distinctName);
    }

    public <T> T lookupEJB2(String earName, String jarName, Class<?> beanClass,
                            Class<T> beanInterface) throws NamingException {
        return lookup(earName, jarName, beanClass, beanInterface, BeanType.EJB2, null);
    }

    public <T> T lookupEJB2(String jarName, Class<?> beanClass, Class<T> beanInterface)
            throws NamingException {
        return lookupEJB2(null, jarName, beanClass, beanInterface);
    }

    public <T> T lookupEntityBean(String earName, String jarName, Class<?> beanClass,
                                  Class<T> beanInterface) throws NamingException {
        return lookup(earName, jarName, beanClass, beanInterface, BeanType.ENTITY, null);
    }

    public <T> T lookupEntityBean(String jarName, Class<?> beanClass, Class<T> beanInterface)
            throws NamingException {
        return lookupEntityBean(null, jarName, beanClass, beanInterface);
    }

    /**
     * Looking for a bean.
     *
     * @param ctx           initial context to be used for searching
     * @param name          name what will be looked for in ctx
     * @param beanInterface interface that the result will be casted to
     * @return beanInterface reference
     * @throws NamingException
     */
    protected <T> T getBean(String name, Class<T> beanInterface) throws NamingException {
        log.info("Looking for: " + name);
        return beanInterface.cast(this.ctx.lookup(name));
    }

    /**
     * Closing initial context without supressing catching {@link NamingException}.
     *
     * @param ctx context that the close() method will be called on
     */
    public void close() {
        try {
            ctx.close();
        } catch (NamingException ne) {
            throw new RuntimeException(ne);
        }
    }

    public InitialContext getInitialContext() {
        return ctx;
    }
}
