/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.ejb.jndi;

import java.util.Properties;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.ejb.client.ContextSelector;
import org.jboss.ejb.client.EJBClientConfiguration;
import org.jboss.ejb.client.EJBClientContext;
import org.jboss.ejb.client.PropertiesBasedEJBClientConfiguration;
import org.jboss.ejb.client.remoting.ConfigBasedEJBClientContextSelector;
import org.jboss.qa.tools.arquillian.extension.EAP6AuthType;
import org.jboss.qa.tools.arquillian.extension.EAP6TestProperties;
import org.jboss.qa.tools.arquillian.extension.EAP6UserAuthConfiguration;

/**
 * @author ochaloup
 */
public class InitialContextDirectoryEAP6EJB extends InitialContextDirectory {
    private static final Logger log = Logger.getLogger(InitialContextDirectoryEAP6EJB.class);
    private ContextSelector<EJBClientContext> previousSelector;
    
    public InitialContextDirectoryEAP6EJB(String hostname, String port) {
        log.info("Using: " + this.getClass().getName() + " on host: " + hostname + ", port: " + port);
        try {
            prepareIntialContext(hostname, port);
        } catch (NamingException ne) {
            throw new RuntimeException(ne);
        }
    }
    
    public InitialContextDirectoryEAP6EJB(String hostname) {
        this(hostname, "4447");
    }
    
    public InitialContextDirectoryEAP6EJB() {
        this("localhost");
    }
    
    private void prepareIntialContext(String hostname, String port) throws NamingException {
        Properties env = new Properties();
        env.put(Context.URL_PKG_PREFIXES, "org.jboss.ejb.client.naming");
        ctx = new InitialContext(env);
        
        Properties prop = new Properties();
        prop.put("remote.connectionprovider.create.options.org.xnio.Options.SSL_ENABLED", "false");
        prop.put("remote.connections", "default");
        prop.put("remote.connection.default.host", hostname);
        prop.put("remote.connection.default.port", port);
        prop.put("remote.connection.default.connect.options.org.xnio.Options.SASL_POLICY_NOANONYMOUS", "false");
        prop.put("remote.connection.default.connect.options.org.xnio.Options.SASL_POLICY_NOPLAINTEXT", "false");
        
        if(EAP6TestProperties.AUTH_TYPE == EAP6AuthType.USER || EAP6TestProperties.AUTH_TYPE == EAP6AuthType.DISABLED) {
            prop.put("remote.connection.default.connect.options.org.xnio.Options.SASL_DISALLOWED_MECHANISMS", "JBOSS-LOCAL-USER");
        }
        if(EAP6TestProperties.AUTH_TYPE == EAP6AuthType.LOCAL) {
            prop.put("remote.connection.default.connect.options.org.xnio.Options.SASL_DISALLOWED_MECHANISMS", "NONE");
        }
        prop.put("remote.connection.default.username", EAP6UserAuthConfiguration.USER_NAME);
        prop.put("remote.connection.default.password", EAP6UserAuthConfiguration.PASSWORD);
        
        final EJBClientConfiguration ejbClientConfiguration = new PropertiesBasedEJBClientConfiguration(prop);
        final ConfigBasedEJBClientContextSelector selector = new ConfigBasedEJBClientContextSelector(ejbClientConfiguration);
        System.out.println("Setting new context selector " + selector);
        previousSelector = EJBClientContext.setSelector(selector);
    }
    
    public <T> String createJndiName(String earName, String jarName, Class<?> beanClass, Class<T> beanInterface, BeanType type, String distinctName) {
        return String.format("ejb:%s/%s/%s/%s!%s%s",
                earName, jarName, distinctName == null ? "" : distinctName, beanClass.getSimpleName(), beanInterface.getName(),
                (type == BeanType.STATEFUL) ? "?stateful" : "");
    }

    @Override
    public <T> T lookup(String earName, String jarName, Class<?> beanClass, Class<T> beanInterface, BeanType beanType, String distinctName)
            throws NamingException {
        earName = earName == null ? "" : earName;
        String name = createJndiName(earName, jarName, beanClass, beanInterface, beanType, distinctName);
        return getBean(name, beanInterface);
    }
    
    @Override
    public void close() {
        EJBClientContext.setSelector(previousSelector);
        super.close();
    }
}
