/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.ejb.jndi;

import org.apache.log4j.Logger;

import java.util.Arrays;

import static org.jboss.qa.ejb.backwardscompatibility.shared.TestHelper.EAP_VERSION_PROPERTY_NAME;
import static org.jboss.qa.ejb.backwardscompatibility.shared.TestHelper.JNDI_LOOKUP_TYPE_PROPERTY_NAME;

/**
 * Creating IntialContextDirectory based on which application server is running.
 * 
 * @author ochaloup
 */
public class InitialContextDirectoryFactory {
    private static final Logger log = Logger.getLogger(InitialContextDirectoryFactory.class);
    

    public static String TESTIP_PROPERTY_NAME = "TESTIP";
    public static int[] EAP_SUPPORTED_VERSIONS = { 5, 6 };

    public static InitialContextDirectory createDirectory() {
        String testIP = System.getProperty(TESTIP_PROPERTY_NAME);
        if(testIP == null || testIP.isEmpty()) {
            testIP = "localhost";
        }
        
        switch (getJndiLookupType()) {
          case EAP5_HTTP:
              return new InitialContextDirectoryEAP5HTTP(testIP);
          case EAP5_JNP:
              return new InitialContextDirectoryEAP5JNP(testIP);
          case EAP6_REMOTE:
              return new InitialContextDirectoryEAP6Remote(testIP);
          case EAP6_EJB_MAPBASED:
              return new InitialContextDirectoryEAP6EJBMapBased(testIP);
          case EAP6_EJB:
          default:
              return new InitialContextDirectoryEAP6EJB(testIP);
        }
    }

    /**
     * Constructing type of initial context directory by system properties.
     */
    private static InitialContextDirectoryType getJndiLookupType() {
        String jndiLookupType = System.getProperty(JNDI_LOOKUP_TYPE_PROPERTY_NAME);
        String eapVersion = System.getProperty(EAP_VERSION_PROPERTY_NAME);

        // Checking EAP version defined
        if (eapVersion == null) {
            throw new IllegalStateException("Property '"
                    + EAP_VERSION_PROPERTY_NAME + "' is not defined");
        }
        Integer numberEapVersion = Integer.valueOf(eapVersion);
        log.debug("Integer value of " + EAP_VERSION_PROPERTY_NAME + " is: " + numberEapVersion);
        Arrays.sort(EAP_SUPPORTED_VERSIONS);
        if (Arrays.binarySearch(EAP_SUPPORTED_VERSIONS, numberEapVersion) < 0) {
            throw new IllegalStateException("EAP version property "
                    + EAP_VERSION_PROPERTY_NAME + " has value " + eapVersion
                    + " whis is not supported. Put value which one from " + Arrays.toString(EAP_SUPPORTED_VERSIONS));
        }

        // no lookup type defined - let's take default ones for appropriate server version
        if(jndiLookupType == null || jndiLookupType.trim().isEmpty()) {
            switch (numberEapVersion) {
              case 5:
                  jndiLookupType = "JNP";
                  break;
              case 6:
                  jndiLookupType = "EJB";
                  break;
            }
        }
        
        return InitialContextDirectoryType.valueOf("EAP" + numberEapVersion + "_" + jndiLookupType.toUpperCase());
    }
}
