/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.tools.arquillian.extension;

import java.io.File;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jboss.arquillian.config.descriptor.api.ArquillianDescriptor;
import org.jboss.arquillian.config.descriptor.api.ContainerDef;
import org.jboss.arquillian.config.descriptor.api.GroupDef;

/**
 * Util functions used in extensions for EAP6.
 * 
 * @author ochaloup
 */
public class ArquillianUtilsEAP6 extends ArquillianUtils {
    private static final Logger log = Logger.getLogger(ArquillianUtilsEAP6.class);

    private static final String EAP6_DEFAULT_MODE = "standalone";
    private static final String EAP6_CONFIGURATION_DIR_NAME = "configuration";
    private static final String EAP6_JBOSS_HOME_PROPERTY_NAME = "jbossHome";
    private static final String EAP6_SERVER_CONFIG_PROPERTY_NAME = "serverConfig";
      
    /**
     * For EAP6 running in standalone mode returns list of filenames of configuration
     * files. It's returned for all groups and all servers defined in arquillian.xml. 
     */
    public static List<File> getConfigFiles(ArquillianDescriptor arqDescriptor) {
        List<File> list = new ArrayList<File>();
        for (GroupDef groupDef : arqDescriptor.getGroups()) {
            for (ContainerDef containerDef : groupDef.getGroupContainers()) {
                File file = getFilename(containerDef);
                if(file != null) {
                    list.add(file);
                }
            }
        }
        for (ContainerDef containerDef : arqDescriptor.getContainers()) {
            File file = getFilename(containerDef);
            if(file != null) {
                list.add(file);
            }            
        }
        return list;
    }
    
    /**
     * This method searches for fileNames at the directories where the config files are placed.
     * 
     * @param fileNames  list of filenames that will be searched at the directories of config files
     * @return  list of files found at the config file dirs
     */
    public static List<File> getFilesInConfigDir(ArquillianDescriptor arqDescriptor, String... fileNames) {
        Set<File> foundFiles = new HashSet<File>();
        List<File> configFiles = ArquillianUtilsEAP6.getConfigFiles(arqDescriptor);
        for(File configFile: configFiles) {
            File dir = configFile.getParentFile();
            for(String fileName: fileNames) {
                String foundFileName = dir.getAbsolutePath() + ArquillianUtils.FS + fileName;
                File foundFile = new File(foundFileName);
                if(foundFile.exists()) {
                    log.debug("Found file " + foundFile + " for config file " + configFile);
                    foundFiles.add(foundFile);
                }
            }
        }
        // remove duplicity
        List<File> returnFiles = new ArrayList<File>();
        returnFiles.addAll(foundFiles);
        return returnFiles;
    }
    
    private static File getFilename(ContainerDef container) {
        String fileName = getConfigFile(container);
        File file = new File(fileName);
        if(file.exists()) {
            return file;
        } else {
            log.warn("[WARN] Configuration file " + fileName + " does not exist! Not adding to process.");
            return null;
        }
    }
    
    /**
     * Returning file path to EAP6 server configuration file for default mode.
     * @param container  arquillian container variable
     * @return  string that is path to config file
     */
    public static String getConfigFile(ContainerDef container) {
        return getEAP6ConfigurationDir(container) + FS +
                container.getContainerProperties().get(EAP6_SERVER_CONFIG_PROPERTY_NAME);
    }
    
    public static String getEAP6ConfigurationDir(ContainerDef container) {
        return  container.getContainerProperties().get(EAP6_JBOSS_HOME_PROPERTY_NAME) + 
                FS + EAP6_DEFAULT_MODE + FS + EAP6_CONFIGURATION_DIR_NAME;
    }
    
    /**
     * Stolen from: org.jboss.crypto.CryptoUtil for not having dependency on it.
     */
    public static String calculateHash(String password) {
        // String hashedPassword=CryptoUtil.createPasswordHash("MD5", "hex",
        // null, null, clearTextPassword);
        byte[] passBytes;
        String passwordHash = null;
        // convert password to byte data
        passBytes = password.getBytes();

        // calculate the hash and apply the encoding.
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(passBytes);
            byte[] hash = md.digest();
            passwordHash = encodeBase16(hash);
        } catch (Exception e) {
            throw new RuntimeException("Password hash calculation failed", e);
        }
        return passwordHash;
    }

    private static String encodeBase16(byte[] bytes) {
        StringBuffer sb = new StringBuffer(bytes.length * 2);
        for (int i = 0; i < bytes.length; i++) {
            byte b = bytes[i];
            // top 4 bits
            char c = (char) ((b >> 4) & 0xf);
            if (c > 9)
                c = (char) ((c - 10) + 'a');
            else
                c = (char) (c + '0');
            sb.append(c);
            // bottom 4 bits
            c = (char) (b & 0xf);
            if (c > 9)
                c = (char) ((c - 10) + 'a');
            else
                c = (char) (c + '0');
            sb.append(c);
        }
        return sb.toString();
    }
}
