/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.ejb.backwardscompatibility;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.naming.NamingException;

import junit.framework.Assert;

import org.apache.log4j.Logger;
import org.jboss.qa.ejb.backwardscompatibility.shared.SendRequestRemote;
import org.jboss.qa.ejb.jndi.InitialContextDirectory;
import org.jboss.qa.ejb.jndi.InitialContextDirectoryFactory;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * @author ochaloup
 */
public abstract class TestCaseParent {

    public static final String EAP6_NODE_NAME = "eap6node";
    protected static InitialContextDirectory ctxDirectory;
    
    // values used in tests for checking correctness of values
    private static final String value = "3";
    private static final String changer = "1";
    private static int valueInt;
    private static int changerInt;
    private static String expectedValue;
    private static int expectedValueInt;

    SendRequestRemote bean;

    private static final Logger log = Logger.getLogger(TestCaseParent.class);
    
    @BeforeClass
    public static void prepare() {
        java.util.logging.Logger.getLogger("org.jboss").setLevel(java.util.logging.Level.ALL);
        java.util.logging.Logger.getLogger("org.xnio").setLevel(java.util.logging.Level.ALL);

        // settings of values that will be sent
        valueInt = Integer.valueOf(value);
        changerInt = Integer.valueOf(changer);
        expectedValue = value + changer;
        expectedValueInt = valueInt + changerInt;
        // getting initial context
        ctxDirectory = InitialContextDirectoryFactory.createDirectory();
    }
    
    @AfterClass
    public static void tearDown() {
        if (ctxDirectory != null) {
            ctxDirectory.close();
        }

    }
    
    /**
     * Method used for getting bean - child class will redefine it for its purposes.
     * @return SendRequestRemote interface
     */
    protected abstract SendRequestRemote getBean() throws NamingException;


    @Test
    public void testInt() throws NamingException {
        log.info("************************ testInt in " + this.getClass().getSimpleName() + "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received primitive type int remote call is not equal to ",
                expectedValueInt, bean.sendInt(valueInt));
        log.info("***** done, success *****");
    }
    
    @Test
    public void testBoolean() throws NamingException {
        log.info("************************ testBoolean in " + this.getClass().getSimpleName() +  "*********************************");
        bean = getBean();
        log.info("***** bean (" + bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        boolean val = true;
        Assert.assertEquals("Received primitive type boolean remote call is not equal to ",
                val, bean.sendBoolean(val));
        log.info("***** done, success *****");
    }

    @Test
    public void testByte() throws NamingException {
        log.info("************************ testByte in " + this.getClass().getSimpleName() + "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received primitive type byte remote call is not equal to ",
                (byte) expectedValueInt, bean.sendByte((byte) valueInt));
        log.info("***** done, success *****");
    }

    @Test
    public void testShort() throws NamingException {
        log.info("************************ testShort in " + this.getClass().getSimpleName() + "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received primitive type short remote call is not equal to ",
                (short) expectedValueInt, bean.sendShort((short) valueInt));
        log.info("***** done, success *****");
    }

    @Test
    public void testLong() throws NamingException {
        log.info("************************ testLong in " + this.getClass().getSimpleName() +  "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received primitive type long remote call is not equal to ",
                (long) expectedValueInt, bean.sendLong((long) valueInt));
        log.info("***** done, success *****");
    }

    @Test
    public void testFloat() throws NamingException {
        log.info("************************ testFloat in " + this.getClass().getSimpleName() +  "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received primitive type float remote call is not equal to ",
                (float) expectedValueInt, bean.sendFloat((float) valueInt));
        log.info("***** done, success *****");
    }

    @Test
    public void testDouble() throws NamingException {
        log.info("************************ testDouble in " + this.getClass().getSimpleName() + "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received primitive type double remote call is not equal to ", 
                (double) expectedValueInt, bean.sendDouble((double) valueInt));
        log.info("***** done, success *****");
    }

    @Test
    public void testChar() throws NamingException {
        log.info("************************ testChar in " + this.getClass().getSimpleName() +  "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received primitive type char remote call is not equal to ",
                (char) expectedValueInt, bean.sendChar((char) valueInt));
        log.info("***** done, success *****");
    }

    @Test
    public void testIntArray() throws NamingException {
        log.info("************************ testIntArray in " + this.getClass().getSimpleName() +  "*********************************");
        bean = getBean();
        int[] arr = {1, 2, 3, 4, 5, 6, 7};
        int[] expectedArr = arr.clone();
        for(int i=0; i<expectedArr.length; i++) {
            expectedArr[i] = expectedArr[i] + changerInt;
        }
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertTrue("Received array of ints from remote call is not equal to ",
                Arrays.equals(expectedArr, bean.sendIntArray(arr)));
        log.info("***** done, success *****");
    }

    @Test
    public void testInteger() throws NamingException {
        log.info("************************ testInteger in " + this.getClass().getSimpleName() +"*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received integer object from remote call is not equal to ",
                (Integer) expectedValueInt, bean.sendInteger(valueInt));
        log.info("***** done, success *****");
    }

    @Test
    public void testCharacter() throws NamingException {
        log.info("************************ testCharacter in " + this.getClass().getSimpleName() +"*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received character object from remote call is not equal to ",
                Character.valueOf((char) expectedValueInt), bean.sendCharacter((char) valueInt));
        log.info("***** done, success *****");
    }

    @Test
    public void testString() throws NamingException {
        log.info("************************ testString in " + this.getClass().getSimpleName() +  "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received string from remote call is not equal to ",
                expectedValue, bean.sendString(value));
        log.info("***** done, success *****");
    }

    @Test
    public void testDate() throws NamingException {
        log.info("************************ testDate in " + this.getClass().getSimpleName() + "*********************************");
        bean = getBean();
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received date value from remote call is not equal to ",
                new Date(expectedValueInt), bean.sendDate(new Date(valueInt)));
        log.info("***** done, success *****");
    }

    @Test
    public void testStringList() throws NamingException {
        log.info("************************ testStringList in " + this.getClass().getSimpleName() + "*********************************");
        bean = getBean();
        List<String> stringList = new ArrayList<String>();
        fillCollection(stringList, false);
        List<String> expectedList = new ArrayList<String>();
        fillCollection(expectedList, true);
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received list of string from remote call is not equal to ",
                expectedList, bean.sendStringList(stringList));
        log.info("***** done, success *****");
    }

    @Test
    public void testStringMap() throws NamingException {
        log.info("************************ testStringMap in " + this.getClass().getSimpleName() + "*********************************");
        bean = getBean();
        Map<Integer, String> stringMap = new HashMap<Integer, String>();
        Map<Integer, String> expectedMap = new HashMap<Integer, String>();
        for(int i=0; i<=10; i++) {
            String str = String.valueOf(i);
            stringMap.put(i, str);
            expectedMap.put(i, str + changer);
        }
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received map of string from remote call is not equal to ",
                expectedMap, bean.sendStringMap(stringMap));
        log.info("***** done, success *****");
    }

    @Test
    public void testStringSet() throws NamingException {
        log.info("************************ testStringSet in " + this.getClass().getSimpleName() +  "*********************************");
        bean = getBean();
        Set<String> stringMap = new HashSet<String>();
        fillCollection(stringMap, false);
        Set<String> expectedMap = new HashSet<String>();
        fillCollection(expectedMap, true);
        log.info("***** bean (" +  bean.toString() + ") acquired, invoking setVariableParam*****");
        bean.setVariableParam(changer);
        log.info("***** invoking*****");
        Assert.assertEquals("Received set of string from remote call is not equal to ",
                expectedMap, bean.sendStringSet(stringMap));
        log.info("***** done, success *****");
    } 
    
    private Collection<String> fillCollection(Collection<String> collectionToFill, boolean isChanger) {
        collectionToFill.clear();
        for(int i=0; i<=10; i++) {
            String val = String.valueOf(i);
            String str = isChanger ? (val + changer) : val;
            collectionToFill.add(str);
        }
        return collectionToFill;
    }
}
