/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.ejb.backwardscompatibility.entitybean;

import java.rmi.RemoteException;
import java.util.Collection;

import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.qa.ejb.jndi.InitialContextDirectory;
import org.jboss.qa.ejb.jndi.InitialContextDirectoryFactory;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * Tests entity bean communication. Inspired from JBoss AS7 testsuite.
 * @author ochaloup 
 */
@RunWith(Arquillian.class)
@RunAsClient
public class EntityBeanTestCase {
    private static final Logger log = Logger.getLogger(EntityBeanTestCase.class);
    private static final String ARCHIVE_NAME = "test-entitybean";
    private static InitialContextDirectory ctxDirectory;

    @Deployment(name = "simple-test", managed=true, testable=false)
    public static Archive<?> deploy() {
        JavaArchive jar = ShrinkWrap.create(JavaArchive.class, ARCHIVE_NAME + ".jar");
        jar.addClasses(TestingEntityHome.class, TestingEntity.class, TestingEntityBean.class, 
                DataStore.class, DataStoreWorker.class, DataStoreWorkerBean.class, DataStoreWorkerHome.class);
        jar.addAsManifestResource("entity/ejb-jar.xml", "ejb-jar.xml");        
        // File testPackage = new File("/tmp", ARCHIVE_NAME + ".jar");
        // jar.as(ZipExporter.class).exportTo(testPackage, true);
        log.debug(jar.toString(true));
        return jar;
    }
    
    @BeforeClass
    public static void prepare() {
        ctxDirectory = InitialContextDirectoryFactory.createDirectory();
    }

    @Test
    public void testSimpleCreate() throws Exception {
        DataStoreWorker dataStore = clearData();
        final TestingEntityHome home = getHome();
        String creatingStr = "Hello";
        final TestingEntity ejbInstance = home.createWithValue(creatingStr);
        final Integer pk = (Integer) ejbInstance.getPrimaryKey();
        Assert.assertEquals("Bean created string '" + creatingStr + "' but it's not in datastore",
            creatingStr, dataStore.getById(pk));
    }

    @Test
    public void testFindByPrimaryKey() throws Exception {
        DataStoreWorker dataStore = clearData();
        final TestingEntityHome home = getHome();
        Integer key = 1099;
        String value = "VALUE" + key;
        dataStore.put(key, value);
        TestingEntity result = home.findByPrimaryKey(key);
        Assert.assertEquals("Datastore contains string '" + value +  "' but bean did not return it.",
            value, result.getValue());
    }

    @Test
    public void testSingleResultFinderMethod() throws Exception {
        DataStoreWorker dataStore = clearData();
        final TestingEntityHome home = getHome();
        Integer key = 1099;
        String value = "VALUE" + key;
        dataStore.put(key, value);
        TestingEntity result = home.findByValue(value);
        Assert.assertEquals("Finder looking for value '" + value + "' but bean did not know about it.",
            value, result.getValue());
        Assert.assertEquals("Finder looking for key '" + key + "' by value but bean did not find it",
            key, result.getPrimaryKey());
    }

    @Test
    public void testCollectionFinderMethod() throws Exception {
        DataStoreWorker dataStore = clearData();
        final TestingEntityHome home = getHome();
        dataStore.put(1000, "Collection");
        dataStore.put(1001, "Collection");
        Collection<TestingEntity> col = home.findCollection();
        for (TestingEntity result : col) {
            Assert.assertEquals("Collection", result.getValue());
        }
    }

    @Test
    public void testRemoveEntityBean() throws Exception {
        DataStoreWorker dataStore = clearData();
        final TestingEntityHome home = getHome();
        Integer key = 42;
        String value = "Remove";
        dataStore.put(key, value);
        TestingEntity result = home.findByPrimaryKey(key);
        Assert.assertEquals(value, result.getValue());
        result.remove();
        Assert.assertFalse(dataStore.containsKey(key));
        try {
            result.getValue();
            Assert.fail("Expected invocation on removed instance to fail");
        } catch (Exception e) {
            log.info("Catching exception: " + e);
            if(!e.getClass().equals(java.rmi.NoSuchObjectException.class) && !e.getCause().getClass().equals(javax.ejb.NoSuchEJBException.class)) {
                Assert.fail("Supposing " + java.rmi.NoSuchObjectException.class + " but " + e + " was caught.");
            }
        }
    }

    private TestingEntityHome getHome() throws NamingException {
        return ctxDirectory.lookupEntityBean(ARCHIVE_NAME, TestingEntityBean.class, TestingEntityHome.class);
    }
    
    private DataStoreWorker getDataStoreWorker() throws NamingException {
        DataStoreWorkerHome home = ctxDirectory.lookupEntityBean(ARCHIVE_NAME, DataStoreWorkerBean.class, DataStoreWorkerHome.class);
        try {
            return home.create();
        } catch (Exception ce) {
            throw new RuntimeException("Error on creating bean "+ DataStoreWorkerBean.class.getName(), ce);
        }
    }
    
    private DataStoreWorker clearData() throws NamingException {
        DataStoreWorker dataStore = getDataStoreWorker();
        try {
            dataStore.clear();
        } catch (RemoteException re) {
            throw new RuntimeException("Error on cleaning datastore", re);
        }
        return dataStore;
    }
}
