/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.ejb.backwardscompatibility.simple;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URLClassLoader;
import java.util.Arrays;

import org.apache.log4j.Logger;
import org.jboss.arquillian.container.test.api.Deployer;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.ejb.client.EJBClientContext;
import org.jboss.qa.ejb.backwardscompatibility.shared.TestHelper;
import org.jboss.qa.ejb.jndi.InitialContextDirectory;
import org.jboss.qa.ejb.jndi.InitialContextDirectoryFactory;
import org.jboss.qa.tools.util.ParameterRule;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.Assume;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * @author ochaloup@redhat.com, jmartisk@redhat.com
 */
@RunWith(Arquillian.class)
@RunAsClient
public class SimpleTestCase {
    private static final Logger log = Logger.getLogger(SimpleTestCase.class);
    private static final String jarName = "simple-test-jar";
    private static final String warName = "simple-test-war";

    private static String archiveName;

    protected static InitialContextDirectory ctxDirectory;

    @Rule
    public ParameterRule rule = new ParameterRule("archiveName", jarName, warName);

    @ArquillianResource
    private Deployer deployer;

    @Deployment(name = jarName, managed = false, testable = false)
    public static Archive<?> createDeploymentJar() {
        log.info("Creating JAR deployment");
        return ShrinkWrap.create(JavaArchive.class, jarName + ".jar")
                .addClasses(SimpleStatelessBeanRemote.class, SimpleStatelessBean.class);
    }

    @Deployment(name = warName, managed = false, testable = false)
    public static Archive<?> createDeploymentWar() throws IOException {
        log.info("Creating WAR deployment");
        File webxml = File.createTempFile("adf", "asdf");
        FileWriter writer = new FileWriter(webxml);
        writer.write("<web-app></web-app>");
        writer.close();
        return ShrinkWrap.create(WebArchive.class, warName + ".war")
                .addClasses(SimpleStatelessBeanRemote.class, SimpleStatelessBean.class)
                .addAsWebInfResource(webxml, "web.xml");
    }

    @Test
    public void doTest() throws Exception {
        java.util.logging.Logger.getLogger("org.jboss").setLevel(java.util.logging.Level.ALL);
        java.util.logging.Logger.getLogger("org.xnio").setLevel(java.util.logging.Level.ALL);
        log.info("*********************************** SimpleTestCase ***********************************************");
        log.info(
                Arrays.toString(((URLClassLoader)EJBClientContext.class.getClassLoader()).getURLs()));
        if(TestHelper.getEAPVersion() == 5)
            Assume.assumeTrue(!archiveName.equals(warName));
        deployer.deploy(archiveName);
        log.info("deployed " + archiveName);
        ctxDirectory = InitialContextDirectoryFactory
                .createDirectory();
        SimpleStatelessBeanRemote bean = ctxDirectory.lookupStateless(archiveName,
                SimpleStatelessBean.class, SimpleStatelessBeanRemote.class);
        log.info("Bean: " + bean);
        String msg;
        try {
            msg = bean.sayHello();
        } catch(Exception e) {
            log.error("ERROR in invocation: " + e.getMessage());
            e.printStackTrace();
            throw e;
        }
        log.info("MSG: " + msg);
        deployer.undeploy(archiveName);
        log.info("undeployed " + archiveName);
    }

    /**
     * Utility class to get name of jar that contains the class.
     */
    private String getPath(Class<?> cls) {
        String cn = cls.getName();
        String rn = cn.replace('.', '/') + ".class";
        String path = getClass().getClassLoader().getResource(rn).getPath();
        int ix = path.indexOf("!");
        if (ix >= 0) {
            return path.substring(0, ix);
        } else {
            return path;
        }
    }

}