/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.tools.arquillian.extension;

import org.apache.log4j.Logger;
import org.jboss.arquillian.config.descriptor.api.ArquillianDescriptor;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.dmr.ModelNode;
import org.junit.Assert;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.*;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
/**
 * This will add user and role to applicatino-properties. 
 * The EJB then will be using these properties for connection.
 * 
 *  AUTH_TYPE.USER - setting user to .properties + SASL_DISALLOWED_MECHANISMS = JBOSS-LOCAL-USER
 *  AUTH_TYPE.LOCAL - no user in .properties + SASL_DISALLOWED_MECHANISMS = NONE
 *  AUTH_TYPE.DISABLED - no user in .properties + SASL_DISALLOWED_MECHANISMS = JBOSS-LOCAL-USER
 * 
 * @author ochaloup
 */
public class EAP6UserAuthConfiguration implements ServerConfigurationDefinition {
    private static final Logger log = Logger.getLogger(EAP6UserAuthConfiguration.class);
    private static final String APPLICATION_REALM_NAME = "ApplicationRealm";
    public static final String USER_NAME = "test";
    public static final String PASSWORD = "password1.";
    private static final String FAKE_ROLE = "ejb";
    public static String HASHED_PASSWORD = null;
    
    /** Filenames that will be changes (used by {@link EAP6BackupAndRestore}) */ 
    public static final String USER_PROPERTIES_FILENAME = "application-users.properties";
    public static final String ROLES_PROPERTIES_FILENAME = "application-roles.properties";
    
    
    @Override
    public void beforeStart(ManagementClient managementClient, ArquillianDescriptor descriptor) {
        if(EAP6TestProperties.AUTH_TYPE == EAP6AuthType.USER) {
            addUsers(descriptor);
            addRoles(descriptor);
        }
    }

    @Override
    public void afterShutdown(ManagementClient managementClient, ArquillianDescriptor descriptor) {
        // nothing to do
    }
    
    @Override
    public void afterStart(ManagementClient managementClient, ArquillianDescriptor descriptor) {
        if(EAP6TestProperties.AUTH_TYPE == EAP6AuthType.DISABLED) {
            disableRemotingSecurityRealm(managementClient);
        }
    }

    @Override
    public void beforeShutdown(ManagementClient managementClient, ArquillianDescriptor descriptor) {
        // nothing to do
    }
    
    /**
     * Creating user and role for properties file used by remoting for autenticate connection.
     * The original content of properties file will be truncated!
     */
    private void addUsers(ArquillianDescriptor descriptor) {
        List<File> userPropertiesFiles = ArquillianUtilsEAP6.getFilesInConfigDir(descriptor, 
                EAP6UserAuthConfiguration.USER_PROPERTIES_FILENAME);
        
        try {
            for (File usersPropertiesFile: userPropertiesFiles) {
                // adding user
                HASHED_PASSWORD = ArquillianUtilsEAP6.calculateHash(USER_NAME + ":" + APPLICATION_REALM_NAME + ":" + PASSWORD);
                PrintWriter propertiesFile = new PrintWriter(usersPropertiesFile);
                propertiesFile.println(USER_NAME + "=" + HASHED_PASSWORD);
                propertiesFile.close();
                log.debug("User properties file " + usersPropertiesFile + " was changed by adding user " + USER_NAME);
            }
        } catch (FileNotFoundException fnfe) {
            throw new RuntimeException("User properties file was not found. The server configuration before the test can't be done.", fnfe);
        }
    }
    
    /**
     * @see {@link EAP6UserAuthConfiguration#addUsers(ArquillianDescriptor)}
     */
    private void addRoles(ArquillianDescriptor descriptor) {
        List<File> rolesPropertiesFiles = ArquillianUtilsEAP6.getFilesInConfigDir(descriptor, 
                EAP6UserAuthConfiguration.ROLES_PROPERTIES_FILENAME);

        try {
            for(File rolesPropertiesFile: rolesPropertiesFiles) {
                PrintWriter propertiesFile = new PrintWriter(rolesPropertiesFile);
                propertiesFile.println(USER_NAME + "=" + FAKE_ROLE);
                propertiesFile.close();
                log.debug("Roles properties file " + rolesPropertiesFile + " was changed by adding role " + FAKE_ROLE + " for user " + USER_NAME);
            }
        } catch (FileNotFoundException fnfe) {
            throw new RuntimeException("Roles properties file was not found. The server configuration before the test can't be done.", fnfe);
        }
    }
    
    private ModelNode reload(ManagementClient managementClient) throws IOException {
        ModelNode operation = new ModelNode();
        operation.get(OP).set("reload");

        return managementClient.getControllerClient().execute(operation);
    }
    
    private void disableRemotingSecurityRealm(ManagementClient managementClient) {
        ModelNode address = new ModelNode();
        address.add(SUBSYSTEM, "remoting");
        address.add("connector", "remoting-connector");

        ModelNode operation = new ModelNode();
        operation.get(OP).set(UNDEFINE_ATTRIBUTE_OPERATION);
        operation.get(OP_ADDR).set(address);
        operation.get("name").set("security-realm");
        
        try {
            ModelNode result = managementClient.getControllerClient().execute(operation);
            Assert.assertEquals("Unsetting the security-realm for remoting was not succesful", SUCCESS, result.get(OUTCOME).asString());
            log.info("ModelNode operation " + UNDEFINE_ATTRIBUTE_OPERATION + " was handled sucessfully with result " + result);
            
            result = reload(managementClient);
            Assert.assertEquals("Reloading operation failed", SUCCESS, result.get(OUTCOME).asString());
            log.info("ModelNode operation reload was handled sucessfully with result " + result);
        } catch (IOException ioe) {
            throw new RuntimeException("Can't unset security-realm on remoting", ioe);
        }
    }
}
