/*
* JBoss, Home of Professional Open Source
* Copyright 2009, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.metadata.spi.scope;

import java.io.Serializable;
import java.util.*;

/**
 * The UnmodifiableScopeKey represents a path which is made up of the path entries.
 *
 * The UnmodifiableScopeKey (Server=Bob,Deployment=Foo.war,Class=Bar)
 * is the child of (Server=Bob,Deployment=Foo.war),
 * which is the child of (Server=Bob).  Think about this statement in terms of
 * a path relationship.  The server (Bob) contains a deployment (Foo.war) and
 * the deployment contains a class (Bar).
 *
 * See historical design notes
 * http://www.jboss.org/index.html?module=bb&op=viewtopic&p=3972233#3972233
 * http://www.jboss.org/index.html?module=bb&op=viewtopic&p=4013747#4013747
 *
 * Thread Safety:  This class is immutable and therefore thread safe.
 *
 * Note: this class requires that class Scope implement Comparable, so that it
 * can correctly sort the array of Scopes.  Note that is not needed for 
 * UnmodifiableScopeKey(ScopeKey), which is already sorted.
 *
 * Warning:  This class (delicately) extends ScopeKey and overides all ScopeKey methods.
 * Since there is no common interface class between the two classes, there are risks to be
 * aware of.  If a new method is added to ScopeKey and not to UnmodifiableScopeKey, undesireable
 * behavior will occur when that new method is invoked (ScopeKey's member variables will be null). 
 *
 * @author <a href="smarlow@redhat.com">Scott Marlow</a>
 * @version $Revision: 75678 $
 */

public final class UnmodifiableScopeKey extends ScopeKey implements Serializable, Cloneable
{
   /** The serialVersionUID */
   private static final long serialVersionUID = -442157853443439820L;

   // the entries in this array are always sorted by (int) Scope.level.level
   // Therefore, the maxScopeLevel is always found at the last element.
   private final Scope theScopes[];

   // Scopes in collection form
   private final Collection<Scope> theScopesCollection;

   // The cached hashCode that is calculated only once by the constructor
   private final int hashcode;

   public UnmodifiableScopeKey( ScopeKey key)
   {
      theScopes = new Scope[key.getScopes().size()];
      key.getScopes().toArray(theScopes);
      // sort(theScopes);  // already sorted
      hashcode = ScopeKey.computeHashCode(theScopes);
      theScopesCollection = Collections.unmodifiableCollection(Arrays.asList(theScopes));
   }

   /**
    * Create a new ScopeKey.
    *
    * @param level the scope level
    * @param qualifier the scope qualifier
    */
   public UnmodifiableScopeKey(ScopeLevel level, Object qualifier)
   {
      this(new Scope(level, qualifier));
   }

   /**
    * Create a new ScopeKey.
    *
    * @param scopes is a collection of Scope instances that make up the represented path
    * @throws IllegalArgumentException if parameter scopes is null
    */
   public UnmodifiableScopeKey(Collection<Scope> scopes)
   {
      if (scopes == null)
         throw new IllegalArgumentException("Null scopes");
      theScopes = new Scope[scopes.size()];
      scopes.toArray(theScopes);
      sort(theScopes);
      hashcode = ScopeKey.computeHashCode(theScopes);
      theScopesCollection = Collections.unmodifiableCollection(Arrays.asList(theScopes));
   }

   /**
    * Create a new ScopeKey.
    *
    * @param scopes is zero or more Scope instances that make up the represented path
    * @throws IllegalArgumentException if parameter scopes is null
    */
   public UnmodifiableScopeKey(Scope... scopes)
   {
      if (scopes == null)
         throw new IllegalArgumentException("Null scopes");
      theScopes = setup(scopes.length, scopes);
      hashcode = ScopeKey.computeHashCode(theScopes);
      theScopesCollection = Collections.unmodifiableCollection(Arrays.asList(theScopes));
   }

   /**
    * Create a new ScopeKey.
    *
    * @param max number of passed Scope instances to use
    * @param scopes is zero or more Scope instances that make up the represented path
    * @throws IllegalArgumentException if parameter scopes is null
    */
   public UnmodifiableScopeKey(int max, Scope... scopes)
   {
      if (scopes == null)
         throw new IllegalArgumentException("Null scopes");
      theScopes = setup(max, scopes);
      hashcode = ScopeKey.computeHashCode(theScopes);
      theScopesCollection = Collections.unmodifiableCollection(Arrays.asList(theScopes));
   }

   public ScopeKey getOptimizedKey()
   {
      return this;
   }

   /**
    * Get the scopes
    *
    * @return the unmodifiable collection of the scopes in expected path order
    */
   public Collection<Scope> getScopes()
   {
      return theScopesCollection;
   }

   /**
    * Get a scope
    *
    * @param level the scope level
    * @return the scope
    * @throws IllegalArgumentException if level is null.
    */
   public Scope getScope(ScopeLevel level)
   {
      if (level == null)
         throw new IllegalArgumentException("Null level");
      // Note that the following search succeeds because we know that the qualifier (empty string)
      // is ignored during the search.
      int idx = Arrays.binarySearch(theScopes, new Scope(level,""));
      if(idx >= 0)
         return theScopes[idx];
      return null;   // not found
   }

   /**
    * Get the maximum scope level
    *
    * @return the largest scope level
    */
   public ScopeLevel getMaxScopeLevel()
   {
      // the maximum scope level is always the last entry (since we are sorted by scopelevel)
      if (theScopes.length > 0)
         return theScopes[theScopes.length - 1].getScopeLevel();
      else
         return null;
   }

   /**
    * Get the parent scope key
    *
    * @return the parent or null if there is no parent (meaning that we
    * are at the top most element in the path)
    */
   public ScopeKey getParent()
   {
    if (theScopes.length < 2)
         return null;
    ScopeKey result = new UnmodifiableScopeKey(theScopes.length - 1, theScopes);
    return result;
   }

   /**
    * Is this parent of key parameter.
    *
    * @param key the key parameter
    * @return true if this is direct parent of key param
    * @throws IllegalArgumentException if parameter key is null
    */
   public boolean isParent(ScopeKey key)
   {
      if (key == null)
         throw new IllegalArgumentException("Null key");

      Scope[] keyArray = key.getArray();

      // The passed key doesn't have a parent
      if (keyArray.length < 2)
         return false;

      // If it is a child, it will have one more scope
      if (theScopes.length != keyArray.length - 1)
         return false;

      for (int looper = 0; looper < keyArray.length - 1; looper ++)
      {
         if(keyArray[looper].equals(theScopes[looper]) == false)
            return false;
      }
      return true;
   }

   /**
    * Get scope for the specified scopeLevel
    *
    * @param scopeLevel the scope level
    * @return the scope or null if there is no such level
    * @throws IllegalArgumentException if parameter scopeLevel is null
    */
   public Scope getScopeLevel(ScopeLevel scopeLevel)
   {
      if (scopeLevel == null)
         throw new IllegalArgumentException("Null scope level");
      return getScope(scopeLevel);
   }

   public String toString()
   {
      return getScopes().toString();
   }

   public boolean equals(Object object)
   {
      if (object == this)
         return true;
      if (object == null || object instanceof ScopeKey == false)
         return false;
      ScopeKey other = (ScopeKey) object;
      Scope[] otherArray = other.getArray();
      return Arrays.equals(theScopes, otherArray);
   }

   public int hashCode()
   {
      return hashcode;
   }

   /**
    * Get the frozen.
    *
    * @return true as UnmodifiableScopeKey is always frozen
    */
   public boolean isFrozen()
   {
      return true;
   }

   /**
    * This method is ignored as UnmodifiableScopeKey is always frozen
    */
   public void freeze()
   {
   }   

   /**
    * Scope cannot be added to an UnmodifiableScopeKey (instead construct a new UnmodifiableScopeKey).  
    * Calling addScope will always fail.
    * 
    * @param scope the scope
    * @throws IllegalArgumentException if scope is null.
    * @throws IllegalStateException because UnmodifiableScopeKey is always frozen
    */
   public Scope addScope(Scope scope)
   {
      if (scope == null)
         throw new IllegalArgumentException("Null scope");
      throw new IllegalStateException("The scope key is frozen");
   }

    /**
    * Scope cannot be added to an UnmodifiableScopeKey (instead construct a new UnmodifiableScopeKey).  
    * Calling addScope will always fail.
    *
    * @param level the scope level
    * @param qualifier the scope qualifier
    * @throws IllegalStateException because UnmodifiableScopeKey is always frozen
    */
   public Scope addScope(ScopeLevel level, Object qualifier)
   {
      throw new IllegalStateException("The scope key is frozen"); 
   }

   /**
    * Scope cannot be removed from an UnmodifiableScopeKey (instead construct a new UnmodifiableScopeKey).  
    * Calling removeScope will always fail.
    *
    * @param scope the scope
    * @throws IllegalStateException because UnmodifiableScopeKey is always frozen
    */
   public Scope removeScope(Scope scope)
   {
      throw new IllegalStateException("The scope key is frozen");
   }

   /**
    * ScopeLevel cannot be removed from an UnmodifiableScopeKey (instead construct a new UnmodifiableScopeKey).  
    * Calling removeScope will always fail.
    *
    * @param scopeLevel the scopeLevel
    * @throws IllegalStateException because UnmodifiableScopeKey is always frozen
    */
   public Scope removeScopeLevel(ScopeLevel scopeLevel)
   {
      throw new IllegalStateException("The scope key is frozen");
   }

   /**
    * clone will always return a frozen copy of the UnmodifiableScopeKey.  
    * This is different then ScopeKey.clone(), which returns an unfrozen ScopeKey.
    */
   public ScopeKey clone()
   {
      ScopeKey result = super.clone();
      return result;
   }

   private static Scope[] setup(int max, Scope... scopes)
   {
      if (max > scopes.length)
         max = scopes.length;
      Scope createdScopes[] = new Scope[max];
      for (int looper = 0; looper < max; looper++)
         createdScopes[looper] = scopes[looper];
      sort(createdScopes);
      return createdScopes;
   }

   private static void sort(Scope scopes[])
   {
      Arrays.sort(scopes);
   }

   protected Scope[] getArray()
   {
      return theScopes;
   }

   protected Collection<Scope> getScopesCollection()
   {
      return theScopesCollection;
   }
}
