/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.metadata.plugins.loader.reflection;

import org.jboss.logging.Logger;
import org.jboss.metadata.plugins.loader.BasicMetaDataLoader;
import org.jboss.metadata.plugins.loader.SimpleMetaDataLoader;
import org.jboss.metadata.spi.retrieval.AnnotationItem;
import org.jboss.metadata.spi.retrieval.AnnotationsItem;
import org.jboss.metadata.spi.retrieval.MetaDataRetrieval;
import org.jboss.metadata.spi.retrieval.simple.SimpleAnnotationItem;
import org.jboss.metadata.spi.retrieval.simple.SimpleAnnotationsItem;
import org.jboss.metadata.spi.scope.CommonLevels;
import org.jboss.metadata.spi.scope.Scope;
import org.jboss.metadata.spi.scope.ScopeKey;
import org.jboss.metadata.spi.scope.UnmodifiableScopeKey;
import org.jboss.metadata.spi.signature.*;
import org.jboss.util.JBossStringBuilder;
import org.jboss.util.Strings;

import java.lang.annotation.Annotation;
import java.lang.reflect.*;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * AnnotatedElementMetaDataLoader.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="cdewolf@redhat.com">Carlo de Wolf</a>
 * @author <a href="ales.justin@jboss.org">Ales Justin</a>
 * @version $Revision: 105213 $
 */
public class AnnotatedElementMetaDataLoader extends BasicMetaDataLoader 
{
   /** The log */
   private static final Logger log = Logger.getLogger(AnnotatedElementMetaDataLoader.class);
   
   /** The annotated element */
   private AnnotatedElement annotated;
   
   private static final ScopeKey getScopeKey(AnnotatedElement annotated)
   {
      Scope scope;
      if (annotated instanceof Class)
      {
         Class<?> clazz = Class.class.cast(annotated);
         scope = new Scope(CommonLevels.CLASS, clazz);
      }
      else if (annotated instanceof Member)
      {
         Member member = (Member) annotated;
         scope = new Scope(CommonLevels.JOINPOINT, member);
      }
      else
      {
         return ScopeKey.DEFAULT_SCOPE;
      }
      return new UnmodifiableScopeKey(scope);
   }
   
   /**
    * Create a new AnnotatedElementMetaDataContext.
    * 
    * @param annotated the annotated element
    */
   public AnnotatedElementMetaDataLoader(AnnotatedElement annotated)
   {
      super(getScopeKey(annotated));
      if (annotated == null)
         throw new IllegalArgumentException("Null annotated element");
      this.annotated = annotated;
   }

   @SuppressWarnings("unchecked")
   public AnnotationsItem retrieveAnnotations()
   {
      Annotation[] annotations = annotated.getAnnotations();
      if (annotations.length == 0)
         return SimpleAnnotationsItem.NO_ANNOTATIONS;
      
      AnnotationItem[] items = new AnnotationItem[annotations.length];
      for (int i = 0; i < items.length; ++i)
         items[i] = new SimpleAnnotationItem(annotations[i]);
      return new SimpleAnnotationsItem(items);
   }
   
   public <T extends Annotation> AnnotationItem<T> retrieveAnnotation(Class<T> annotationType)
   {
      T annotation = annotated.getAnnotation(annotationType);
      if (annotation == null)
         return null;
      return new SimpleAnnotationItem<T>(annotation);
   }

   public MetaDataRetrieval getComponentMetaDataRetrieval(Signature signature)
   {
      if (signature == null)
         return null;

      if (annotated instanceof Class)
      {
         Class<?> clazz = Class.class.cast(annotated);
         if (signature instanceof ConstructorSignature)
         {
            ConstructorSignature constructorSignature = (ConstructorSignature) signature;
            Constructor<?> constructor = constructorSignature.getConstructor();
            if (constructor == null)
               constructor = SecurityActions.findConstructor(clazz, signature.getParametersTypes(clazz));
            if (constructor == null)
            {
               if (log.isTraceEnabled())
                  log.trace("Constructor with signature " + signature + " does not exist on class " + clazz.getName());
               return null;
            }
            if (constructor.getAnnotations().length == 0)
               return null;
            return new AnnotatedElementMetaDataLoader(constructor);
         }
         else if (signature instanceof MethodSignature)
         {
            MethodSignature methodSignature = (MethodSignature) signature;
            Method method = methodSignature.getMethod();
            if (method == null)
               method = SecurityActions.findMethod(clazz, signature.getName(), signature.getParametersTypes(clazz));
            if (method == null)
            {
               if (log.isTraceEnabled())
                  log.trace("Method with signature " + signature + " does not exist on class " + clazz.getName());
               return null;
            }
            if (method.getAnnotations().length == 0)
            {
               if (method.isBridge())
               {
                  method = searchForRealBridgeMethodSignature(method); 
                  if (method == null)
                     return null;
               }
               else
                  return null;
            }
            return new AnnotatedElementMetaDataLoader(method);
         }
         else if (signature instanceof DeclaredMethodSignature)
         {
            DeclaredMethodSignature methodSignature = (DeclaredMethodSignature) signature;
            Method method = methodSignature.getMethod();
            if (method == null)
            {
               clazz = getDeclaringClass(clazz, methodSignature.getDeclaringClass());
               if (clazz == null)
                  return null;
               try
               {
                  method = SecurityActions.findDeclaredMethod(clazz, signature.getName(), signature.getParametersTypes(clazz));
               }
               catch (NoSuchMethodException nsme)
               {
                  if (log.isTraceEnabled())
                     log.trace("Method with signature " + signature + " does not exist on class " + clazz.getName());
                  return null;     
               }
            }
            if (method.getAnnotations().length == 0)
            {
               if (method.isBridge())
               {
                  method = searchForRealBridgeMethodSignature(method);
                  if (method == null)
                     return null;
               }
               else
                  return null;
            }
            return new AnnotatedElementMetaDataLoader(method);
         }
         else if (signature instanceof MethodParametersSignature)
         {
            MethodParametersSignature methodParametersSignature = (MethodParametersSignature) signature;
            Method method = methodParametersSignature.getMethod();
            if (method == null)
               method = SecurityActions.findMethod(clazz, signature.getName(), signature.getParametersTypes(clazz));
            if (method == null)
            {
               if (log.isTraceEnabled())
                  log.trace("Method with signature " + signature + " does not exist on class " + clazz.getName());
               return null;
            }
            Annotation[][] paramAnnotations = method.getParameterAnnotations();
            if (paramAnnotations[methodParametersSignature.getParam()].length == 0)
               return null;
            return new SimpleMetaDataLoader(paramAnnotations[methodParametersSignature.getParam()]);
         }
         else if (signature instanceof ConstructorParametersSignature)
         {
            ConstructorParametersSignature constructorParametersSignature = (ConstructorParametersSignature) signature;
            Constructor<?> constructor = constructorParametersSignature.getConstructor();
            if (constructor == null)
               constructor = SecurityActions.findConstructor(clazz, signature.getParametersTypes(clazz));
            if (constructor == null)
            {
               if (log.isTraceEnabled())
                  log.trace("Constructor with signature " + signature + " does not exist on class " + clazz.getName());
               return null;
            }
            Annotation[][] paramAnnotations = constructor.getParameterAnnotations();
            if (paramAnnotations[constructorParametersSignature.getParam()].length == 0)
               return null;
            return new SimpleMetaDataLoader(paramAnnotations[constructorParametersSignature.getParam()]);
         }
         else if (signature instanceof FieldSignature)
         {
            FieldSignature fieldSignature = (FieldSignature) signature;
            Field field = fieldSignature.getField();
            if (field == null)
               field = SecurityActions.findField(clazz, signature.getName());
            if (field == null)
            {
               if (log.isTraceEnabled())
                  log.trace("Field " + signature.getName() + " does not exist on class " + clazz.getName());
               return null;
            }
            if (field.getAnnotations().length == 0)
               return null;
            return new AnnotatedElementMetaDataLoader(field);
         }
      }

      if (annotated instanceof Method)
      {
         if (signature instanceof MethodParametersSignature)
         {
            Method method = (Method)annotated;
            Annotation[][] paramAnnotations = method.getParameterAnnotations();
            MethodParametersSignature sig = (MethodParametersSignature) signature;
            if (paramAnnotations[sig.getParam()].length == 0)
               return null;
            return new SimpleMetaDataLoader(paramAnnotations[sig.getParam()]);
         }
      }

      if (annotated instanceof Constructor)
      {
         if (signature instanceof ConstructorParametersSignature)
         {
            Constructor<?> constructor = Constructor.class.cast(annotated);
            Annotation[][] paramAnnotations = constructor.getParameterAnnotations();
            ConstructorParametersSignature sig = (ConstructorParametersSignature) signature;
            if (paramAnnotations[sig.getParam()].length == 0)
               return null;
            return new SimpleMetaDataLoader(paramAnnotations[sig.getParam()]);
         }
      }

      return null;
   }

   public boolean isEmpty()
   {
      Annotation[] annotations = annotated.getAnnotations();
      return annotations == null || annotations.length == 0;
   }

   public String toString()
   {
      JBossStringBuilder buffer = new JBossStringBuilder();
      Strings.defaultToString(buffer, this);
      buffer.append("[");
      buffer.append(annotated);
      buffer.append("]");
      return buffer.toString();
   }

   /**
    * Get the declaring class from the reference class
    * 
    * @param reference the reference class
    * @param declaringClass the declaring class
    * @return the class or null if the declaring class is not a super class of the reference
    */
   private static Class<?> getDeclaringClass(Class<?> reference, String declaringClass)
   {
      while (reference != null)
      {
         if (declaringClass.equals(reference.getName()))
            return reference;
         
         reference = reference.getSuperclass();
      }
      return null;
   }

   /**
    * Search for real bridge method.
    *
    * @param bridge the current method
    * @return real bridge method
    */
   private Method searchForRealBridgeMethodSignature(Method bridge)
   {
      Class<?> declaringClass = bridge.getDeclaringClass();
      Class<?>[] parameters = bridge.getParameterTypes();

      List<Method> matching = new ArrayList<Method>();
      Method[] all = declaringClass.getDeclaredMethods();
      for (Method m : all)
      {
         if (m.getName().equals(bridge.getName()) &&
               m.getParameterTypes().length == parameters.length &&
               m.equals(bridge) == false &&
               m.isBridge() == false)
            matching.add(m);
      }
      
      if (matching.size() == 1)
         return matching.get(0);
      
      //Should not happen
      if (matching.size() == 0)
         throw new IllegalStateException("No original methods found: " + bridge);
      
      for (Iterator<Method> it = matching.iterator() ; it.hasNext() ; )
      {
         Method cur = it.next();
         
         if (bridge.getReturnType().isAssignableFrom(cur.getReturnType()) == false)
         {
            it.remove();
            continue;
         }

         Class<?>[] currentParameters = cur.getParameterTypes();
         for (int i = 0 ; i < parameters.length ; i++)
         {
            if (parameters[i].isAssignableFrom(currentParameters[i]) == false)
            {
               it.remove();
               break;
            }
         }
      }

      if (matching.size() == 1)
         return matching.get(0);
       
      //Should not happen
      if (matching.size() == 0)
         throw new IllegalStateException("No original methods found: " + bridge);

      if (log.isTraceEnabled())
         log.trace("Could not determine original method for " + bridge + " found: " + matching);
      
      return null;
   }
}
