/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist;

import java.util.Arrays;
import java.util.Stack;

import org.jboss.reflect.spi.TypeInfo;

import javassist.CtClass;
import javassist.NotFoundException;
import javassist.bytecode.BadBytecode;
import javassist.bytecode.SignatureAttribute;
import javassist.bytecode.SignatureAttribute.ClassSignature;
import javassist.bytecode.SignatureAttribute.ClassType;
import javassist.bytecode.SignatureAttribute.ObjectType;
import javassist.bytecode.SignatureAttribute.TypeArgument;
import javassist.bytecode.SignatureAttribute.TypeParameter;
import javassist.bytecode.SignatureAttribute.TypeVariable;

/**
 * TODO This is prototype code! Once there are some tests for things like nested generics etc., it is very likely that we will need
 * to change signatures of methods in this class as needed.
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class JavassistHelper
{
   /**
    * Looking at the classes between clazz and search, determine the type of the type parameter with the passed in index
    * 
    * @param clazz the sub class
    * @param search the parent class or interface we are searching for
    * @param parameter the index of the type parameter we are looking for
    * @return the type
    */
   static String determineType(CtClass clazz, CtClass search, int parameter)
   {
      Stack<CtClass> hierarchy = new Stack<CtClass>();      
      try
      {
         determineHierarchy(hierarchy, clazz, search);
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException(e);
      }
      return determineType(hierarchy, parameter);
   }

   static int determineInfoIndex(TypeInfo[] actualTypeArguments, CtClass clazz, CtClass search, int parameter)
   {
      Stack<CtClass> hierarchy = new Stack<CtClass>();      
      try
      {
         determineHierarchy(hierarchy, clazz, search);
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException(e);
      }
      return determineInfoIndex(actualTypeArguments, hierarchy, parameter);
   }
   
   /**
    * Determine the type of the parameter in the top-level class. 
    * 
    * @param hierarchy the hierarchy of classes as determined by {@link JavassistHelper#determineHierarchy(Stack, CtClass, CtClass)}
    * @param parameter the index of the parameter
    * @return the type
    */
   private static String determineType(Stack<CtClass> hierarchy, int parameter)
   {
      TypeDecider decider = new TypeDecider();
      decider.determineType(hierarchy, parameter);
      return decider.name;
   }
   
   private static int determineInfoIndex(TypeInfo[] actualTypeArguments, Stack<CtClass> hierarchy, int parameter)
   {
      TypeDecider decider = new TypeDecider();
      decider.determineType(hierarchy, parameter);
      return decider.lastIndex;
   }

   /**
    * Gets the ClassSignature for a class
    * 
    * @param clazz the CtClass
    * @return the ClassSignature
    */
   static ClassSignature getClassSignature(CtClass clazz)
   {
      if (clazz == null)
         throw new IllegalArgumentException("Null clazz");

      SignatureAttribute signature = (SignatureAttribute)clazz.getClassFile().getAttribute(SignatureAttribute.tag);
      if (signature == null)
         return null;
      String sig = signature.getSignature();
   
      try
      {
         return SignatureAttribute.toClassSignature(sig);
      }
      catch (BadBytecode e)
      {
         throw new IllegalStateException(e);
      }
   }

   /**
    * Figures out the path between the passed in classes
    * 
    * @param hierarchy receives the CtClasses that make up the hierarchy
    * @param current the sub class
    * @param search the parent class or interface we are searching for
    * @throws IllegalArgumentException if any of the parameters are null
    */
   private static boolean determineHierarchy(Stack<CtClass> hierarchy, CtClass current, CtClass search) throws NotFoundException
   {
      if (hierarchy == null)
         throw new IllegalArgumentException("Null hierarchy");
      if (current == null)
         throw new IllegalArgumentException("Null current");
      if (search == null)
         throw new IllegalArgumentException("Null search");
   
      if (current == null)
         return false;
      
      hierarchy.push(current);
      
      if (current.equals(search))
         return true;
      
      CtClass[] interfaces = current.getInterfaces();
      if (search.isInterface() && interfaces != null)
      {
         for (int i = 0 ; i < interfaces.length ; i++)
         {
            boolean result = determineHierarchy(hierarchy, interfaces[i], search);
            if (result)
               return true;
         }
      }
      
      CtClass superClass = current.getSuperclass();
      boolean result = determineHierarchy(hierarchy, superClass, search);
      if (result)
         return true;
      
      hierarchy.pop();
      return false;
   }
   
   /**
    * Finds the TypeArgument used for the parent class/interface
    * 
    * @param parent the parent class or interface
    * @param classSig the signature of the "current" class, i.e. the child of <code>parent</code>/
    * @param index the index of the type parameter in the parent class
    * @return the found TypeArgument
    * @throws IllegalArgumentException if the index is greater than the length of TypeArguments found in the classSig, or if any of the parameters are null  
    */
   private static TypeArgument findSuperClassOrInterfaceArguments(CtClass parent, ClassSignature classSig, int index)
   {
      if (parent == null)
         throw new IllegalArgumentException();
      TypeArgument[] arguments = null; 
      if (parent.isInterface())
      {
         ClassType[] types = classSig.getInterfaces();
         for (int i = 0 ; i < types.length ; i++)
         {
            if (types[i].getName().equals(parent.getName()))
            {
               arguments = types[i].getTypeArguments();
               break;
            }
         }
         if (arguments == null)
            throw new IllegalStateException("Could not find " + parent.getName() + " in " + Arrays.toString(types));
      }
      else
      {
         arguments =  classSig.getSuperClass().getTypeArguments();
      }
      
      if (arguments.length <= index)
         throw new IllegalArgumentException("Argument " + index + " requested, but only " + arguments.length + " exist");
      return arguments[index];
   }


   private static class TypeDecider
   {
      int lastIndex;
      CtClass last = null;
      String name;
      
      private void determineType(Stack<CtClass> hierarchy, int parameter)
      {
         //TODO This should maybe return the full ObjectType instead
         CtClass clazz = null;
         
         TypeParameter targetType = null;
         lastIndex = parameter;
         while (true)
         {
            if (hierarchy.empty())
               break;
            last = clazz;
            clazz = hierarchy.pop();
            
            ClassSignature classSig = getClassSignature(clazz);
            if (classSig != null)
            {
               TypeParameter[] typeParameters = classSig.getParameters();
      
               if (last == null)
               {
                  if (typeParameters.length <= parameter)
                     throw new IllegalArgumentException("Parameter " + parameter + " requested, but only " + typeParameters.length + " exist.");
                  targetType = typeParameters[parameter];
               }
               else
               {
                  TypeArgument argument = findSuperClassOrInterfaceArguments(last, classSig, lastIndex);
                  ObjectType type = argument.getType();
                  if (type == null)
                     continue;
                  if (type instanceof ClassType)
                  {
                     name = ((ClassType) type).getName();
                     return;
                  }
                     
                  String name = null; 
                  if (type instanceof TypeVariable)
                     name= ((TypeVariable)type).getName();

                  for (int i = 0 ; i < typeParameters.length ; i++)
                  {
                     if (typeParameters[i].getName().equals(name))
                     {
                        lastIndex = i;
                        targetType = typeParameters[i];
                        break;
                     }
                  }
               }
            }
            else
            {
               break;
            }
         }
         if (targetType != null)
         {
            //TODO also check interfaces
             name = ((ClassType)targetType.getClassBound()).getName();
             return;
         }
      }
   }
}
