/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode;

import java.lang.annotation.Annotation;
import java.lang.reflect.Modifier;

import javassist.bytecode.BadBytecode;
import javassist.bytecode.SignatureAttribute;
import javassist.bytecode.SignatureAttribute.MethodSignature;

import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.plugins.bytecode.bytes.BehaviourBytes;
import org.jboss.reflect.spi.AnnotationInfo;
import org.jboss.reflect.spi.AnnotationValue;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.ParameterInfo;
import org.jboss.reflect.spi.TypeInfo;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 105060 $
 */
public abstract class BytecodeBehaviourInfo extends BytecodeAnnotatedInfo
{
   /** The serialVersionUID */
   private static final long serialVersionUID = -494071110672611729L;

   /** The parameters */
   protected transient volatile ParameterInfo[] parameters;
   
   /** The parameter types */
   protected transient volatile TypeInfo[] parameterTypes;
   
   /** The exception types */
   protected transient volatile ClassInfo[] exceptionTypes;
   
   /** The type info */
   protected final BytecodeTypeInfo typeInfo;
   
   /** The constructor or method */
   protected final BehaviourBytes behaviour;

   private transient volatile boolean initializedMethodSignature;
   
   private transient volatile MethodSignature methodSignature;

   public BytecodeBehaviourInfo(AnnotationHelper annotationHelper, BytecodeTypeInfo typeInfo, BehaviourBytes behaviour)
   {
      super(annotationHelper);
      this.typeInfo = typeInfo;
      this.behaviour = behaviour;
   }

   public int getModifiers()
   {
      return behaviour.getModifiers(); 
   }

   public boolean isPublic()
   {
      return Modifier.isPublic(getModifiers());
   }

   public boolean isStatic()
   {
      return Modifier.isStatic(getModifiers());
   }

   public boolean isVolatile()
   {
      return Modifier.isVolatile(getModifiers());
   }

   public BytecodeTypeInfo getDeclaringClass()
   {
      return typeInfo;
   }

   public ClassInfo[] getExceptionTypes()
   {
      if (exceptionTypes == null)
      {
         String[] types = behaviour.getExceptionTypeTypeInfoNames();
         ClassInfo[] exceptionTypes = new ClassInfo[types.length];
         for (int i = 0; i < types.length; ++i)
         {
            try
            {
               exceptionTypes[i] = (ClassInfo) typeInfo.getFactory().getTypeInfo(types[i], getDeclaringClass().getClassLoaderInternal());
            }
            catch (ClassNotFoundException e)
            {
               throw new RuntimeException(e);
            }
         }
         this.exceptionTypes = exceptionTypes;
      }
      return exceptionTypes;
   }

   protected void generateParameters()
   {
      if (parameters != null)
         return;
      
      SignatureKey key = getSignatureKey();
      MethodSignature sig = getMethodSignature();
      if (sig != null && sig.getParameterTypes().length == key.getParams().length)
      {
         SignatureAttribute.Type[] types = sig.getParameterTypes();
         TypeInfo[] parameterTypes = new TypeInfo[types.length];
         for (int i = 0 ; i < types.length ; i++)
            parameterTypes[i] = typeInfo.getFactory().getTypeInfo(typeInfo.getClassLoaderInternal(), types[i], BytecodeTypeVariableSpy.createForBehavior(typeInfo.getClassSignature(), sig));
         this.parameterTypes = parameterTypes;
      }
      else
      {
         TypeInfo[] parameterTypes = new TypeInfo[key.getParams().length];
         for (int i = 0 ; i < parameterTypes.length ; i++)
         {
            try
            {
               parameterTypes[i] = typeInfo.getFactory().getTypeInfo(key.getParams()[i], typeInfo.getClassLoaderInternal());
            }
            catch (ClassNotFoundException e)
            {
               throw new RuntimeException(e);
            }
         }
         this.parameterTypes = parameterTypes;
      }
      
      ParameterInfo[] parameters = new ParameterInfo[parameterTypes.length];
      for (int i = 0; i < parameterTypes.length; ++i)
         parameters[i] = new BytecodeParameterInfo(annotationHelper, this, i, parameterTypes[i]);
      this.parameters = parameters;
   }

   public ParameterInfo[] getParameters()
   {
      if (parameters == null)
         generateParameters();
      return parameters;
   }

   public TypeInfo[] getParameterTypes()
   {
      if (parameterTypes == null)
         generateParameters();
      return parameterTypes;
   }

   public AnnotationValue[] getAnnotations()
   {
      return getAnnotations(behaviour);
   }
   
   public abstract String getName();
      
   protected void setupParameterAnnotations(Object[][] annotations)
   {
      if (annotations == null)
         return;
      for (int param = 0 ; param < annotations.length ; param++)
      {
         AnnotationValue[] annotationValues = new AnnotationValue[annotations[param].length];
         for (int ann = 0 ; ann < annotationValues.length ; ann++)
         {
            Class<?> clazz = ((Annotation)annotations[param][ann]).annotationType();

            AnnotationInfo info = (AnnotationInfo)((BytecodeTypeInfoFactoryImpl)annotationHelper).getTypeInfo(clazz);
            annotationValues[ann] = annotationHelper.createAnnotationValue(info, annotations[param][ann]);
         }
         ((BytecodeParameterInfo)parameters[param]).setAnnotations(annotationValues);
      }
   }
   
   protected void createParameterAnnotations()
   {
      Object[][] parameterAnnotations = behaviour.getParameterAnnotations();
      setupParameterAnnotations(parameterAnnotations);
   }

   protected MethodSignature getMethodSignature()
   {
      if (!initializedMethodSignature)
      {
         try
         {
            String sig = behaviour.getGenericSignature();
            if (sig != null)
               methodSignature = SignatureAttribute.toMethodSignature(sig);
         }
         catch (BadBytecode e)
         {
            throw new RuntimeException(e);
         }
         initializedMethodSignature = true;
      }
      return methodSignature;
   }

   public SignatureKey getSignatureKey()
   {
      return behaviour.getSignatureKey();
   }
   
   public String getDescriptor()
   {
      return behaviour.getJvmSignature();
   }
}
