/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.bytes.asm;

import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import org.jboss.reflect.plugins.bytecode.bytes.BytecodePrimitive;
import org.jboss.reflect.plugins.bytecode.bytes.ClassBytes;
import org.jboss.reflect.util.objectweb.asm.AnnotationVisitor;
import org.jboss.reflect.util.objectweb.asm.Attribute;
import org.jboss.reflect.util.objectweb.asm.ClassReader;
import org.jboss.reflect.util.objectweb.asm.ClassVisitor;
import org.jboss.reflect.util.objectweb.asm.FieldVisitor;
import org.jboss.reflect.util.objectweb.asm.Label;
import org.jboss.reflect.util.objectweb.asm.MethodVisitor;
import org.jboss.reflect.util.objectweb.asm.Type;
import org.jboss.reflect.util.objectweb.asm.commons.EmptyVisitor;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class Util
{
   final static Annotation[] NO_ANNOTATIONS = new Annotation[0];
   
   private final static Map<Class<?>, Map<String, String>> RETURN_TYPES_BY_NAME = Collections.synchronizedMap(new WeakHashMap<Class<?>, Map<String, String>>());
   private final static Map<Class<?>, Map<String, Object>> DEFAULT_VALUES = Collections.synchronizedMap(new WeakHashMap<Class<?>, Map<String, Object>>()); 
   
   static String jvmNameToTypeInfoName(String jvmName)
   {
      if (jvmName == null)
         return null;
      return jvmName.replace('/', '.');
   }
   
   static AnnotationVisitor createAnnotationVisitor(ParentReader parent, ClassLoader loader, String desc)
   {
      return new AnnotationReader(parent, loader, desc);
   }
   
   private static Class<?> loadClass(ClassLoader loader, String desc)
   {
      try
      {
         BytecodePrimitive primitive = BytecodePrimitive.valueOf(desc);
         if (primitive != null)
            return primitive.getPrimitiveClass();

         return SecurityActions.loadClass(loader, toClassName(desc).replace('/', '.'));
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException("Could not load class '" + toClassName(desc) + "' with loader " + loader);
      }
   }
   
   private static Object handleSpecial(ClassLoader loader, Object value)
   {
      if (value.getClass() == Type.class)
      {
         String desc = ((Type)value).getDescriptor();
         int arrayDims = 0;
         while (desc.charAt(arrayDims) == '[')
            arrayDims++;
            
         Class<?> clazz = loadClass(loader, desc.substring(arrayDims));
         if (arrayDims == 0)
            return clazz;
         int[] dims = new int[arrayDims];
         return Array.newInstance(clazz, dims).getClass();
      }
      return value;
   }
   
   private static String toClassName(String desc)
   {
      if (desc.charAt(0) != 'L' && desc.length() <= 2 && desc.charAt(desc.length() - 1) != ';')
         throw new IllegalArgumentException("Not a proper descriptor " + desc);
      return desc.substring(1, desc.length() - 1);
   }

   interface ParentReader
   {
      void setValueInParent(String name, Object value);
   }

   private static class AnnotationReader implements AnnotationVisitor, ParentReader
   {
      Map<String, Object> values = new HashMap<String, Object>();
      
      final ParentReader parent;
      
      final ClassLoader loader;
      
      Class<?> clazz;
      
      ClassBytes classBytes;
      
      Map<String, String> returnTypesByName;
      
      AnnotationReader(ParentReader parent, ClassLoader loader, String desc)
      {
         this.parent = parent;
         this.loader = loader;
         
         clazz = loadClass(loader, desc);
         classBytes = AsmClassBytesFactory.INSTANCE.loadClassBytes(loader, toClassName(desc));
         if (classBytes == null)
            throw new IllegalStateException("Could not load bytes for " + toClassName(desc) + " in " + loader);
         
         returnTypesByName = RETURN_TYPES_BY_NAME.get(clazz);
         if (returnTypesByName == null)
         {
            AllAnnotationAttributesReader allAnnotationAttributesReader = new AllAnnotationAttributesReader();
            ((AsmClassBytes)classBytes).getReader().accept(allAnnotationAttributesReader, AsmClassBytes.STANDARD_FLAGS | ClassReader.INCLUDE_METHOD_OVERVIEW);
            returnTypesByName = Collections.unmodifiableMap(allAnnotationAttributesReader.returnTypesByName);
            RETURN_TYPES_BY_NAME.put(clazz, returnTypesByName);
         }
      }

      public void setValueInParent(String name, Object value)
      {
         values.put(name, value);
      }

      public void visit(String name, Object value)
      {
         values.put(name, handleSpecial(loader, value));
      }
   
      public AnnotationVisitor visitAnnotation(String name, String desc)
      {
         return new NestedAnnotationReader(this, loader, name, desc);
      }
   
      public AnnotationVisitor visitArray(String name)
      {
         return new ArrayReader(this, loader, name, loadClass(loader, returnTypesByName.get(name).substring(1)));
      }

      Object createAnnotation()
      {
         try
         {
            return  AnnotationProxy.createProxy(values, clazz);
         }
         catch (Exception e)
         {
            throw new RuntimeException("Error creating annotation proxy for " + clazz.getName());
         }
            
      }
      
      void handleDefaultAttributes()
      {
         Map<String, Object> defaults = null;
         for (String name : returnTypesByName.keySet())
         {
            if (!values.containsKey(name))
            {
               if (defaults == null)
               {
                  defaults = DEFAULT_VALUES.get(clazz);
                  if (defaults == null)
                  {
                     if (classBytes instanceof AsmClassBytes)
                     {
                        DefaultAnnotationAttributeReader defaultsReader = new DefaultAnnotationAttributeReader(loader, returnTypesByName);
                        ((AsmClassBytes)classBytes).getReader().accept(defaultsReader, AsmClassBytes.STANDARD_FLAGS | ClassReader.INCLUDE_METHOD_OVERVIEW | ClassReader.INCLUDE_DEFAULT_ANNOTATION_VALUES);
                        defaults = defaultsReader.defaultAttributesByName;
                     }
                     else
                        defaults = Collections.<String, Object>emptyMap();
                  }
                  DEFAULT_VALUES.put(clazz, Collections.unmodifiableMap(defaults));
               }
               Object value = defaults.get(name);
               if (value == null)
                  throw new IllegalStateException("No value and no default for " + name + " in " + clazz.getName());
               values.put(name, value);
            }
         }
      }
      
      public void visitEnd()
      {
         handleDefaultAttributes();
         parent.setValueInParent(null, createAnnotation());
      }
      
      @SuppressWarnings("unchecked")
      public void visitEnum(String name, String desc, String value)
      {
         Class<?> enumClass = loadClass(loader, desc);
         values.put(name, Enum.valueOf((Class<? extends Enum>)enumClass, value));
      }

      private static class AllAnnotationAttributesReader extends EmptyVisitor
      {
         Map<String, String> returnTypesByName = new HashMap<String, String>();

         @Override
         public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions, int byteCodeIndex)
         {
            if (name.charAt(0) != '<' && desc.charAt(1) == ')')
            {
               returnTypesByName.put(name, desc.substring(2));
            }
            return null;
         }
      }
      
      private static class DefaultAnnotationAttributeReader extends EmptyVisitor
      {
         
         final Map<String, Object> defaultAttributesByName = new HashMap<String, Object>();
         final ClassLoader loader;
         final Map<String, String> returnTypesByName;
         
         public DefaultAnnotationAttributeReader(ClassLoader loader, Map<String, String> returnTypesByName)
         {
            this.loader = loader;
            this.returnTypesByName = returnTypesByName;
         }
         
         @Override
         public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions, int byteCodeIndex)
         {
            return new AnnotationDefaultReader(name);
         }

//         @Override
//         public void visitEnd()
//         {
//            super.visitEnd();
//         }

         private class AnnotationDefaultReader extends EmptyMethodVisitor implements AnnotationVisitor, ParentReader
         {
            String name;
            
            private AnnotationDefaultReader(String name)
            {
               this.name = name;
            }
            
            @Override
            public AnnotationVisitor visitAnnotationDefault()
            {
               return this;
            }

            public void setValueInParent(String name, Object value)
            {
               defaultAttributesByName.put(this.name, value);
            }

            public void visit(String name, Object value)
            {
               defaultAttributesByName.put(this.name, handleSpecial(loader, value));
            }

            public AnnotationVisitor visitAnnotation(String name, String desc)
            {
               return new NestedAnnotationReader(this, loader, this.name, desc);
            }
         
            public AnnotationVisitor visitArray(String name)
            {
               return new ArrayReader(this, loader, this.name, loadClass(loader, returnTypesByName.get(this.name).substring(1)));
            }

            @SuppressWarnings("unchecked")
            public void visitEnum(String name, String desc, String value)
            {
               Class<?> enumClass = loadClass(loader, desc);
               defaultAttributesByName.put(this.name, Enum.valueOf((Class<? extends Enum>)enumClass, value));
            }
         }
      }
   }   
   
   private static class NestedAnnotationReader extends AnnotationReader
   {
      private final String name;
      
      NestedAnnotationReader(ParentReader parent, ClassLoader loader, String name, String desc)
      {
         super(parent, loader, desc);
         this.name = name;
      }

      @Override
      public void visitEnd()
      {
         handleDefaultAttributes();
         parent.setValueInParent(name, createAnnotation());
      }
   }
   
   private static class ArrayReader implements AnnotationVisitor, ParentReader
   {
      final ParentReader parent;
      final ClassLoader loader;
      final String name;
      final Class<?> type;
      final List<Object> values = new ArrayList<Object>(); 
      
      ArrayReader(ParentReader parent, ClassLoader loader, String name, Class<?> type)
      {
         this.parent = parent;
         this.name = name;
         this.type = type; 
         this.loader = loader;
      }
      
      public void visit(String name, Object value)
      {
         values.add(handleSpecial(loader, value));
      }

      public AnnotationVisitor visitAnnotation(String name, String desc)
      {
         //return new NestedAnnotationReader(this, factory, loader, name, desc);
         return new AnnotationReader(this, loader, desc);
      }

      public AnnotationVisitor visitArray(String name)
      {
         throw new IllegalArgumentException("Can't have nested arrays");
      }

      public void visitEnd()
      {
         Object[] src = values.toArray();
         Object[] dest = (Object[])Array.newInstance(type, values.size());
         for (int i = 0 ; i < src.length ; i++)
            dest[i] = type.cast(src[i]);
         //System.arraycopy(src, 0, dest, 0, src.length);
         
         parent.setValueInParent(name, dest);
      }

      @SuppressWarnings("unchecked")
      public void visitEnum(String name, String desc, String value)
      {
         Class<?> enumClass = loadClass(loader, desc);
         values.add(Enum.valueOf((Class<? extends Enum>)enumClass, value));         
      }

      public void setValueInParent(String name, Object value)
      {
         values.add(value);
      }
      
   }
   
   static abstract class EmptyClassVisitor implements ClassVisitor
   {

      public void visit(int version, int access, String name, String signature, String superName, String[] interfaces)
      {
      }

      public AnnotationVisitor visitAnnotation(String desc, boolean visible)
      {
         return null;
      }

      public void visitAttribute(Attribute attr)
      {
      }

      public void visitEnd()
      {
      }

      public FieldVisitor visitField(int access, String name, String desc, String signature, Object value, int index)
      {
         return null;
      }

      public void visitInnerClass(String name, String outerName, String innerName, int access)
      {
      }

      public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions, int byteCodeIndex)
      {
         return null;
      }

      public void visitOuterClass(String owner, String name, String desc)
      {
      }

      public void visitSource(String source, String debug)
      {
      }
   }
   
   static class EmptyFieldVisitor implements FieldVisitor
   {

      public AnnotationVisitor visitAnnotation(String desc, boolean visible)
      {
         return null;
      }

      public void visitAttribute(Attribute attr)
      {
      }

      public void visitEnd()
      {
      }
      
   }
   
   static class EmptyMethodVisitor implements MethodVisitor
   {

      public AnnotationVisitor visitAnnotation(String desc, boolean visible)
      {
         return null;
      }

      public AnnotationVisitor visitAnnotationDefault()
      {
         return null;
      }

      public void visitAttribute(Attribute attr)
      {
      }

      public void visitCode()
      {
      }

      public void visitEnd()
      {
      }

      public void visitFieldInsn(int opcode, String owner, String name, String desc)
      {
      }

      public void visitFrame(int type, int nLocal, Object[] local, int nStack, Object[] stack)
      {
      }

      public void visitIincInsn(int var, int increment)
      {
      }

      public void visitInsn(int opcode)
      {
      }

      public void visitIntInsn(int opcode, int operand)
      {
      }

      public void visitJumpInsn(int opcode, Label label)
      {
      }

      public void visitLabel(Label label)
      {
      }

      public void visitLdcInsn(Object cst)
      {
      }

      public void visitLineNumber(int line, Label start)
      {
      }

      public void visitLocalVariable(String name, String desc, String signature, Label start, Label end, int index)
      {
      }

      public void visitLookupSwitchInsn(Label dflt, int[] keys, Label[] labels)
      {
      }

      public void visitMaxs(int maxStack, int maxLocals)
      {
      }

      public void visitMethodInsn(int opcode, String owner, String name, String desc)
      {
      }

      public void visitMultiANewArrayInsn(String desc, int dims)
      {
      }

      public AnnotationVisitor visitParameterAnnotation(int parameter, String desc, boolean visible)
      {
         return null;
      }

      public void visitTableSwitchInsn(int min, int max, Label dflt, Label[] labels)
      {
      }

      public void visitTryCatchBlock(Label start, Label end, Label handler, String type)
      {
      }

      public void visitTypeInsn(int opcode, String type)
      {
      }

      public void visitVarInsn(int opcode, int var)
      {
      }
      
   }
}
