/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.bytecode;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import javassist.NotFoundException;
import javassist.bytecode.BadBytecode;
import javassist.bytecode.SignatureAttribute;
import javassist.bytecode.SignatureAttribute.ClassSignature;
import javassist.bytecode.SignatureAttribute.ClassType;

import org.jboss.reflect.plugins.ClassInfoImpl;
import org.jboss.reflect.plugins.PackageInfoImpl;
import org.jboss.reflect.plugins.TypeInfoAttachments;
import org.jboss.reflect.plugins.ValueConvertor;
import org.jboss.reflect.plugins.bytecode.bytes.ClassBytes;
import org.jboss.reflect.plugins.bytecode.bytes.ConstructorBytes;
import org.jboss.reflect.plugins.bytecode.bytes.FieldBytes;
import org.jboss.reflect.plugins.bytecode.bytes.MethodBytes;
import org.jboss.reflect.spi.AnnotationValue;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.ConstructorInfo;
import org.jboss.reflect.spi.FieldInfo;
import org.jboss.reflect.spi.InterfaceInfo;
import org.jboss.reflect.spi.MethodInfo;
import org.jboss.reflect.spi.PackageInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.reflect.spi.TypeInfoFactory;
import org.jboss.util.JBossStringBuilder;

/**
 * BytecodeTypeInfo.
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * 
 * @version $Revision: 105060 $
 */
public class BytecodeTypeInfo extends BytecodeInheritableAnnotationHolder implements ClassInfo, InterfaceInfo, BytecodeClassInfo
{
   /** The serialVersionUID */
   private static final long serialVersionUID = -5072033691434335775L;

   /** The get classloader permission */
   protected static final RuntimePermission GET_CLASSLOADER_PERMISSION = new RuntimePermission("getClassLoader");
   
   static final BytecodeTypeInfo COLLECTION, MAP;
   static
   {
      BytecodeTypeInfoFactory factory = new BytecodeTypeInfoFactory();
      try
      {
         COLLECTION = (BytecodeTypeInfo)factory.getTypeInfo(Collection.class);
         MAP = (BytecodeTypeInfo)factory.getTypeInfo(Map.class);
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }
   
   private final static Map<String, BytecodeFieldInfo> EMPTY_FIELDS = Collections.emptyMap(); 
   
   private final static Map<SignatureKey, MethodInfo> EMPTY_METHODS = Collections.emptyMap();
   
   private final Map<SignatureKey, List<MethodInfo>> EMPTY_VOLATILE_METHODS = Collections.emptyMap(); 
   

   /** The factory */
   private BytecodeTypeInfoFactoryImpl factory;

   /** The name */
   private final String name;
   
   /** The class */
   protected volatile Class<? extends Object> clazz;

   private final static Map<SignatureKey, BytecodeConstructorInfo> EMPTY_CONSTRUCTORS = Collections.emptyMap();
   
   /** The constructors */
   private volatile Map<SignatureKey, BytecodeConstructorInfo> constructors = EMPTY_CONSTRUCTORS;;

   /** The constructors */
   private volatile ConstructorInfo[] constructorArray;

   /** The fields */
   private volatile Map<String, BytecodeFieldInfo> fields = EMPTY_FIELDS;

   /** The fields */
   private volatile FieldInfo[] fieldArray;

   /** The methods */
   private volatile Map<SignatureKey, MethodInfo> methods = EMPTY_METHODS;
   
   private volatile Map<SignatureKey, List<MethodInfo>> volatileMethods = EMPTY_VOLATILE_METHODS; 

   /** The methods */
   private volatile MethodInfo[] methodArray;

   /** The package info */
   private volatile PackageInfo packageInfo;
   
   /** The attachments */
   private transient volatile TypeInfoAttachments attachments;
   
   /** The generic super class */
   private volatile ClassInfo genericSuperClass = ClassInfoImpl.UNKNOWN_CLASS;

   /** The generic interfaces */
   private volatile InterfaceInfo[] genericInterfaces = ClassInfoImpl.UNKNOWN_INTERFACES;

   /** The component type if this is a collection */
   private volatile TypeInfo componentType = ClassInfoImpl.UNKNOWN_TYPE;
   
   /** The key type if this is a Map */
   private volatile TypeInfo keyType = ClassInfoImpl.UNKNOWN_TYPE;
   
   /** The value type if this is a Map */
   private volatile TypeInfo valueType = ClassInfoImpl.UNKNOWN_TYPE;
   
   /** Whether we are a map */
   private volatile Boolean isMap;
   
   /** Whether we are a map */
   private volatile Boolean isCollection;

   private transient volatile ClassInfo superClass = ClassInfoImpl.UNKNOWN_CLASS;

   private transient volatile InterfaceInfo[] interfaces = ClassInfoImpl.UNKNOWN_INTERFACES;
   
   private transient volatile boolean initializedClassSignature;
   
   private transient volatile ClassSignature classSignature;


   /**
    * Create a new BytecodeTypeInfo.
    * 
    * @param factory the factory
    * @param classBytes the class
    * @param clazz the class
    * @param name the name
    */
   BytecodeTypeInfo(BytecodeTypeInfoFactoryImpl factory, String name, ClassBytes classBytes, Class<? extends Object> clazz)
   {
      super(classBytes, factory);
      if (factory == null)
         throw new IllegalArgumentException("Null factory");
      if (name == null)
         throw new IllegalArgumentException("Null name");
      this.factory = factory;
      this.clazz = clazz;
      this.name = name;
   }

   public String getName()
   {
      return name;
   }

   public boolean isInterface()
   {
      return classBytes.isInterface();
   }

   public String getSimpleName()
   {
      String name = getName();
      int index = name.lastIndexOf('.');
      if (index < 0)
         return name;
      else
         return name.substring(index + 1);
   }

   public int getModifiers()
   {
      return classBytes.getModifiers();
   }

   public boolean isPublic()
   {
      return Modifier.isPublic(getModifiers());
   }

   public boolean isStatic()
   {
      return Modifier.isStatic(getModifiers());
   }

   public boolean isVolatile()
   {
      return Modifier.isVolatile(getModifiers());
   }

   @Deprecated
   public Class<? extends Object> getType()
   {
      try
      {
         if(clazz == null)
            clazz = SecurityActions.loadClass(classBytes.getClassLoader(), getName());
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException(e);
      }
      
      return clazz;
   }

   public ClassLoader getClassLoader()
   {
      // looks like Javassist ClassPool::getClassLoader
      // doesn't check for security, so we should
      SecurityManager sm = System.getSecurityManager();
      if (sm != null)
         sm.checkPermission(GET_CLASSLOADER_PERMISSION);
      
      return getClassLoaderInternal();
   }

   public ClassLoader getClassLoaderInternal()
   {
      return classBytes.getClassLoader();
   }
   
   public ClassInfo getSuperclass()
   {
      if (superClass == ClassInfoImpl.UNKNOWN_CLASS)
      {
         if (isInterface())
            this.superClass = null;
         else
         {
            String superClass = classBytes.getSuperClassTypeInfoName();
            if (superClass == null)
               this.superClass = null;
            else
            {
               try
               {
                  this.superClass = (ClassInfo)factory.getTypeInfo(superClass, getClassLoaderInternal());
               }
               catch (ClassNotFoundException e)
               {
                  throw new RuntimeException("Error loading super class '" + superClass + 
                  		"' for " + getName() + " in " + getClassLoaderInternal());
               }
            }
         }
      }
      return superClass;
   }

   public ClassInfo getGenericSuperclass()
   {
      if (genericSuperClass == ClassInfoImpl.UNKNOWN_CLASS)
      {
         ClassSignature classSig = getClassSignature();

         if (getClassSignature() != null)
         {
            ClassType type = classSig.getSuperClass();
            genericSuperClass = (ClassInfo)factory.getTypeInfo(getClassLoaderInternal(), type, BytecodeTypeVariableSpy.createForClass(classSig));
         }
         else
         {
            genericSuperClass = getSuperclass();
         }
      }
         
      return genericSuperClass;   
   }

   public InterfaceInfo[] getInterfaces()
   {
      if (interfaces == ClassInfoImpl.UNKNOWN_INTERFACES)
      {
         String[] ifaces = classBytes.getInterfaceTypeInfoNames();
         if (ifaces != null && ifaces.length > 0)
         {
            InterfaceInfo[] result = new InterfaceInfo[ifaces.length];
            for (int i = 0 ; i < result.length ; i++)
            {
               try
               {
                  result[i] = (InterfaceInfo)factory.getTypeInfo(ifaces[i], getClassLoaderInternal());
               }
               catch (ClassNotFoundException e)
               {
                  throw new RuntimeException("Error loading interface '" + superClass + 
                        "' for " + getName() + " in " + getClassLoaderInternal());
               }
            }
            interfaces = result;
         }
         else
         {
            interfaces = null;
         }
      }
      return interfaces;
   }

   public InterfaceInfo[] getGenericInterfaces()
   {
      InterfaceInfo[] interfaces = getInterfaces();
      if (interfaces == null || interfaces.length == 0)
      {
         genericInterfaces = null;
      }
      if (genericInterfaces == ClassInfoImpl.UNKNOWN_INTERFACES)
      {
         InterfaceInfo[] infos = new InterfaceInfo[getInterfaces().length];
         
         ClassSignature classSig = getClassSignature();

         if (classSig != null)
         {
            ClassType[] types = classSig.getInterfaces();
            for (int i = 0 ; i < types.length ; i++)
            {
               ClassInfo info = (ClassInfo)factory.getTypeInfo(getClassLoaderInternal(), types[i], BytecodeTypeVariableSpy.createForClass(classSig));
               if (info instanceof InterfaceInfo == false)
                  throw new IllegalStateException(info + " is not an InterfaceInfo");
               infos[i] = (InterfaceInfo)info; 
            }         
         
            genericInterfaces = infos;
         }
         else
         {
            genericInterfaces = getInterfaces();
         }
      }
      return genericInterfaces;
   }

   public ConstructorInfo[] getDeclaredConstructors()
   {
      if (constructorArray == null)
      {
         ConstructorBytes[] declaredConstructors = classBytes.getDeclaredConstructorBytes();
         if (declaredConstructors.length == 0)
            constructorArray = ClassInfoImpl.UNKNOWN_CONSTRUCTORS;
         else
         {
            for (int i = 0 ; i < declaredConstructors.length ; i++)
               generateConstructorInfo(declaredConstructors[i]);
            
            synchronized (constructors)
            {
               Collection<BytecodeConstructorInfo> constructorCollection = constructors.values();
               constructorArray = constructorCollection.toArray(new ConstructorInfo[constructorCollection.size()]);
            }
         }
      }
      return constructorArray;
   }

   public ConstructorInfo getDeclaredConstructor()
   {
      return getDeclaredConstructor(new SignatureKey(null));
   }
   
   public ConstructorInfo getDeclaredConstructor(String... parameters) throws ClassNotFoundException
   {
      TypeInfo[] typeParams = new TypeInfo[parameters.length];
      for(int i=0; i<parameters.length;i++)
      {
         typeParams[i] = factory.getTypeInfo(parameters[i], getClassLoaderInternal());
      }
      return getDeclaredConstructor(typeParams);
   }
   
   public ConstructorInfo getDeclaredConstructor(TypeInfo... parameters)
   {
      return getDeclaredConstructor(new SignatureKey(null, parameters));
   }

   private ConstructorInfo getDeclaredConstructor(SignatureKey key)
   {
      synchronized (constructors)
      {
         ConstructorInfo constructor = constructors.get(key);
         if (constructor != null)
            return constructor;
      }
      if (constructorArray != null)
         return null;
      return generateConstructorInfo(key);
   }
   
   public FieldInfo getDeclaredField(String fieldName)
   {
      synchronized (fields)
      {
         FieldInfo field = fields.get(fieldName);
         if (field != null)
            return field;
      }
      if (fieldArray != null)
         return null;

      FieldBytes[] fields = classBytes.getDeclaredFieldBytes();
      for (int i = 0 ; i < fields.length ; i++)
      {
         if (fields[i].getName().equals(fieldName))
            return generateFieldInfo(fields[i]);
      }
      return null;
   }

   public FieldInfo[] getDeclaredFields()
   {
      if (fieldArray == null)
      {
         FieldBytes[] declaredFields = classBytes.getDeclaredFieldBytes();
         if (declaredFields == null || declaredFields.length == 0)
            fieldArray = ClassInfoImpl.UNKNOWN_FIELDS;
         else
         {
            synchronized (fields)
            {
               for (int i = 0; i < declaredFields.length; ++i)
                  generateFieldInfo(declaredFields[i]);
               Collection<BytecodeFieldInfo> fieldCollection = fields.values();
               fieldArray = fieldCollection.toArray(new FieldInfo[fieldCollection.size()]);
               
            }
         }
      }
      return fieldArray;
   }
   
   public MethodInfo getDeclaredMethod(String methodName)
   {
      return getDeclaredMethod(new SignatureKey(methodName));
   }

   public MethodInfo getDeclaredMethod(String methodName, TypeInfo... parameters)
   {
      return getDeclaredMethod(new SignatureKey(methodName, parameters));
   }

   public MethodInfo getDeclaredMethod(String methodName, String... parameters) throws ClassNotFoundException
   {
      TypeInfo[] typeParams = new TypeInfo[parameters.length];
      for(int i=0; i<parameters.length;i++)
      {
         typeParams[i] = factory.getTypeInfo(parameters[i], classBytes.getClassLoader());
      }         
      return getDeclaredMethod(methodName, typeParams);
   }
   
   private MethodInfo getDeclaredMethod(SignatureKey key)
   {
      synchronized (methods)
      {
         MethodInfo method = methods.get(key);
         if (method != null)
            return method;
      }
      if (methodArray != null)
         return null;
      return generateMethodInfo(key);
   }
   
   public MethodInfo[] getDeclaredMethods()
   {
      if (methodArray == null)
      {
         MethodBytes[] declaredMethods = classBytes.getDeclaredMethodBytes();
         if (declaredMethods.length == 0)
            methodArray = ClassInfoImpl.UNKNOWN_METHODS;
         else
         {
            synchronized (methods)
            {
               for (int i = 0; i < declaredMethods.length; ++i)
                  generateMethodInfo(declaredMethods[i]);
               Collection<MethodInfo> methodCollection = methods.values();
               
               if (volatileMethods.size() > 0)
               {
                  Collection<MethodInfo> allMethods = new ArrayList<MethodInfo>();
                  allMethods.addAll(methodCollection);
                  methodCollection = allMethods;
                  
                  for (List<MethodInfo> infos : volatileMethods.values())
                  {
                     if (infos.size() > 0)
                        methodCollection.addAll(infos);
                  }
               }
               
               methodArray = methodCollection.toArray(new MethodInfo[methodCollection.size()]);
            }
         }
         
      }
      return methodArray;
   }

   public boolean isArray()
   {
      return false;
   }

   public boolean isCollection()
   {
      if (isCollection == null)
      {
         try
         {
            isCollection = BytecodeGenericsHelper.determineHierarchy(null, this, COLLECTION);
         }
         catch (NotFoundException e)
         {
            throw new IllegalStateException(e);
         }
      }
      return isCollection;
   }

   public boolean isMap()
   {
      if (isMap == null)
      {
         try
         {
            isMap = BytecodeGenericsHelper.determineHierarchy(null, this, MAP);
         }
         catch (NotFoundException e)
         {
            throw new IllegalStateException(e);
         }
      }
      return isMap;
   }
   
   public boolean isAnnotation()
   {
      return classBytes.isAnnotation();
   }

   public boolean isEnum()
   {
      return classBytes.isEnum();
   }

   public boolean isPrimitive()
   {
      return false;
   }

   /**
    * Get an array class
    * 
    * @param clazz the class
    * @return the array class
    */
   public static Class<?> getArrayClass(Class<?> clazz)
   {
      return Array.newInstance(clazz, 0).getClass();
   }

   public TypeInfo getArrayType()
   {
      Class<?> arrayClass = getArrayClass(getType());
      return factory.getTypeInfo(arrayClass);
   }

   @SuppressWarnings("deprecation")
   public Object newArrayInstance(int size) throws Throwable
   {
      if (isArray() == false)
         throw new ClassCastException(this + " is not an array.");
      return Array.newInstance(getComponentType().getType(), size);
   }

   @SuppressWarnings("deprecation")
   public boolean isAssignableFrom(TypeInfo info)
   {
      if (info == null)
         throw new NullPointerException("Parameter info cannot be null!");

      return getType().isAssignableFrom(info.getType());
   }

   public boolean isInstance(Object object)
   {
      return getType().isInstance(object);
   }

   public TypeInfoFactory getTypeInfoFactory()
   {
      return factory;
   }

   public Object convertValue(Object value) throws Throwable
   {
      return ValueConvertor.convertValue(getType(), value);
   }

   public Object convertValue(Object value, boolean replaceProperties) throws Throwable
   {
      return ValueConvertor.convertValue(getType(), value, replaceProperties);
   }

   public Object convertValue(Object value, boolean replaceProperties, boolean trim) throws Throwable
   {
      return ValueConvertor.convertValue(getType(), value, replaceProperties, trim);
   }

   @Override
   protected int getHashCode()
   {
      return getName().hashCode();
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof TypeInfo == false)
         return false;

      TypeInfo other = (TypeInfo) obj;
      return getName().equals(other.getName());
   }

   @Override
   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(getName());
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      buffer.append("name=").append(getName());
      super.toString(buffer);
   }

   /**
    * Get the factory
    * 
    * @return the factory
    */
   public BytecodeTypeInfoFactoryImpl getFactory()
   {
      return factory;
   }

   /**
    * Generate constructor info
    * 
    * @param constructor the constructor
    * @return the constructor info
    */
   protected ConstructorInfo generateConstructorInfo(ConstructorBytes constructor)
   {
      BytecodeConstructorInfo info = new BytecodeConstructorInfo(factory, this, constructor);
      synchronized (constructors)
      {
         if (constructors == EMPTY_CONSTRUCTORS)
            constructors = new HashMap<SignatureKey, BytecodeConstructorInfo>(1);
         constructors.put(constructor.getSignatureKey(), info);
      }
      return info;
   }

   /**
    * Generate constructor info
    * 
    * @param key the key
    * @return the constructor info
    */
   protected ConstructorInfo generateConstructorInfo(SignatureKey key)
   {
      ConstructorBytes[] constructors = classBytes.getDeclaredConstructorBytes();
      for (int i = 0 ; i < constructors.length ; i++)
      {
         if (constructors[i].getSignatureKey().equals(key))
            return generateConstructorInfo(constructors[i]);
      }
      return null;
   }

   /**
    * Generate field info
    * 
    * @param field the field
    * @return the field info
    */
   protected FieldInfo generateFieldInfo(FieldBytes field)
   {
      BytecodeFieldInfo info = new BytecodeFieldInfo(factory, this, field);
      synchronized (fields)
      {
         if (fields == EMPTY_FIELDS)
            fields = new HashMap<String, BytecodeFieldInfo>(1);
         fields.put(field.getName(), info);
      }
      return info;
   }

   /**
    * Generate method info
    * 
    * @param key the key
    * @return the method info
    */
   protected MethodInfo generateMethodInfo(SignatureKey key)
   {
      MethodBytes[] methods = classBytes.getDeclaredMethodBytes();
      for (int i = 0 ; i < methods.length ; i++)
      {
         if (methods[i].getName().equals(key.getName()) && methods[i].getSignatureKey().equals(key))
            return generateMethodInfo(methods[i]);
      }
      return null;
   }

   /**
    * Generate method info
    * 
    * @param method the method
    * @return the method info
    */
   protected MethodInfo generateMethodInfo(MethodBytes method)
   {
      MethodInfo info = new BytecodeMethodInfo(factory, this, method);
      synchronized (methods)
      {
         if (Modifier.isVolatile(method.getModifiers()))
         {
            List<MethodInfo> infos = volatileMethods.get(method.getSignatureKey());
            if (infos == null)
            {
               infos = new CopyOnWriteArrayList<MethodInfo>();
               if (volatileMethods == EMPTY_VOLATILE_METHODS)
                  volatileMethods = new HashMap<SignatureKey, List<MethodInfo>>(1);
               volatileMethods.put(method.getSignatureKey(), infos);
            }
            infos.add(info);
         }
         else
         {
            if (methods == EMPTY_METHODS)
               methods = new HashMap<SignatureKey, MethodInfo>(1);
            methods.put(method.getSignatureKey(), info);
         }
      }
      return info;
   }


   @Override
   public AnnotationValue[] getAnnotations()
   {
      return getAnnotations(classBytes);
   }

   @Override
   public ClassInfo getSuperHolder()
   {
      return getSuperclass();
   }

   public TypeInfo[] getActualTypeArguments()
   {
      return ClassInfoImpl.UNKNOWN_TYPES;
   }

   public TypeInfo getOwnerType()
   {
      return null;
   }

   public ClassInfo getRawType()
   {
      return this;
   }

   public TypeInfo getComponentType()
   {
      if (!isCollection())
         return null;
      
      if (componentType == ClassInfoImpl.UNKNOWN_TYPE)
      {
         componentType = findTypeInfo(COLLECTION, 0);
      }
      return componentType;
   }

   public TypeInfo getKeyType()
   {
      if (!isMap())
         return null;
      
      if (keyType == ClassInfoImpl.UNKNOWN_TYPE)
      {
         keyType = findTypeInfo(MAP, 0);
      }
      return keyType;
   }

   public TypeInfo getValueType()
   {
      if (!isMap())
         return null;
      
      if (valueType == ClassInfoImpl.UNKNOWN_TYPE)
      {
         valueType = findTypeInfo(MAP, 1);
      }
      return valueType;
   }

   private TypeInfo findTypeInfo(ClassInfo iface, int parameter)
   {
      try
      {
         ClassType type = BytecodeGenericsHelper.determineType(this, iface, parameter);
         return factory.getTypeInfo(getClassLoaderInternal(), type, BytecodeTypeVariableSpy.createForClass(getClassSignature()));
      }
      catch (Exception e1)
      {
         throw new RuntimeException(e1);
      }
   }
   
   public PackageInfo getPackage()
   {
      if (packageInfo == null)
      {
         String packageName = getPackageName();
         if (packageName != null)
         {
            AnnotationValue[] annotations = null;
            ClassLoader cl = getClassLoaderInternal();

            String name = packageName + ".package-info";
            ClassBytes clazz = factory.loadClassBytes(cl, name);
            try
            {
               ClassInfo info = (ClassInfo)factory.get(clazz, null, name, cl, false);
               annotations = info.getAnnotations();
            }
            catch (ClassNotFoundException e)
            {
            }
            
            packageInfo = new PackageInfoImpl(packageName, annotations);
         }
      }
      return packageInfo;
   }

   protected String getPackageName()
   {
      return getPackageName(getName());
   }
   
   protected String getPackageName(String classname)
   {
      int index = classname.lastIndexOf('.');
      return index < 0 ? null : classname.substring(0, index);
   }
   
   public void setAttachment(String name, Object attachment)
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      synchronized (this)
      {
         if (attachments == null)
         {
            if (attachment == null)
               return;
            attachments = new TypeInfoAttachments();;
         }
      }
      if (attachment == null)
         attachments.removeAttachment(name);
      else
         attachments.addAttachment(name, attachment);
   }

   public <T> T getAttachment(Class<T> expectedType)
   {
      if (expectedType == null)
         throw new IllegalArgumentException("Null expectedType");
      Object result = getAttachment(expectedType.getName());
      if (result == null)
         return null;
      return expectedType.cast(result);
   }

   public Object getAttachment(String attachmentName)
   {
      if (attachmentName == null)
         throw new IllegalArgumentException("Null name");
      synchronized (this)
      {
         if (attachments == null)
            return null;
      }
      return attachments.getAttachment(attachmentName);
   }
   
   protected Object writeReplace()
   {
      return new MarshalledClassInfo(getType());
   }
   
   public static class MarshalledClassInfo implements Serializable
   {
      private static final long serialVersionUID = 1L;
      
      Class<?> type;

      public MarshalledClassInfo(Class<?> type)
      {
         this.type = type;
      }
      
      protected Object readResolve()
      {
         TypeInfoFactory tif = new BytecodeTypeInfoFactory();
         return tif.getTypeInfo(type);
      }
   }

   public String getTypeVariable()
   {
      return null;
   }
   
   public ClassSignature getClassSignature()
   {
      if (!initializedClassSignature)
      {
         try
         {
            String sig = classBytes.getGenericSignature();
            if (sig != null)
               classSignature = SignatureAttribute.toClassSignature(sig);
         }
         catch (BadBytecode e)
         {
            throw new RuntimeException(e);
         }
         initializedClassSignature = true;
      }
      return classSignature;
   }
   
   public ClassBytes getClassBytes()
   {
      return classBytes;
   }
}
