/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.bytecode;

import java.util.Arrays;

import javassist.bytecode.Descriptor;

import org.jboss.reflect.plugins.bytecode.bytes.BehaviourBytes;
import org.jboss.reflect.plugins.bytecode.bytes.BytecodePrimitive;
import org.jboss.reflect.plugins.bytecode.bytes.FieldBytes;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.JBossStringBuilder;


/**
 * SignatureKey.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="kabir.khann@jboss.com">Kabir Khan</a>
 * @version $Revision$
 */
public class SignatureKey
{
   /** The name */
   private final String name;
   
   /** The parameter names */
   private final String[] params;
   
   private volatile String descriptor;
   
   /** The cached hashcode */
   private transient int cachedHashCode = Integer.MIN_VALUE;
   
   private static final String[] NO_PARAMS = new String[0];
   
   /**
    * Create a new SignatureKey.
    * 
    * @param name the name
    * @param typeInfos the type infos
    */
   public SignatureKey(String name)
   {
      this.name = name;
      this.params = NO_PARAMS;
   }
   
   /**
    * Create a new SignatureKey.
    * 
    * @param name the name
    * @param typeInfos the type infos
    */
   public SignatureKey(String name, TypeInfo... typeInfos)
   {
      this.name = name;
      if (typeInfos != null && typeInfos.length > 0)
      {
         params = new String[typeInfos.length];
         for (int i = 0; i < typeInfos.length; ++i)
            params[i] = typeInfos[i].getName();
      }
      else
         params = NO_PARAMS;
   }
   
   /**
    * Create a new SignatureKey.
    * 
    * @param name the name
    * @param behavior
    */
   public SignatureKey(String name, BehaviourBytes behavior)
   {
      this.name = name;
      params = getParameterTypeStringsForSignature(behavior);
   }
   
   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof SignatureKey == false)
         return false;
      
      SignatureKey other = (SignatureKey) obj;
      
      if (name == null && other.name != null)
         return false;
      if (name != null && other.name == null)
         return false;
      if (name != null && name.equals(other.name) == false)
         return false;
      
      if (params == null && other.params == null)
         return true;
      if (params == null && other.params != null)
         return false;
      if (params != null && other.params == null)
         return false;
      
      if (params.length != other.params.length)
         return false;
      
      for (int i = 0; i < params.length; ++i)
      {
         if (params[i].equals(other.params[i]) == false)
            return false;
      }
      return true;
   }
   
   @Override
   public int hashCode()
   {
      if (cachedHashCode == Integer.MIN_VALUE)
      {
         JBossStringBuilder builder = new JBossStringBuilder();
         if (name != null)
            builder.append(name);
         if (params != null)
         {
            for (int i = 0; i < params.length; ++i)
               builder.append(params[i]);
         }
         cachedHashCode = builder.toString().hashCode();
      }
      return cachedHashCode;
   }
   
   @Override
   public String toString()
   {
      if (params == null)
         return name + "[]";
      return name + Arrays.asList(params);
   }
   
   public String getName()
   {
      return name;
   }
   
   /**
    * Constructs the parameter strings for a behaviour's signature in the same way
    * as TypeInfo does
    * 
    * @param behaviour
    * @return the parameter strings
    */
   private String[] getParameterTypeStringsForSignature(BehaviourBytes behavior)
   {
      descriptor = behavior.getJvmSignature();
      String[] args = new String[Descriptor.numOfParameters(descriptor)];

      int n = 0;
      int i = 1;
      do
      {
         i = getParameterTypeString(descriptor, i, args, n++);
      }
      while (i > 0 && n < args.length);
      
      
      return args;
   }

   public static String getReturnType(BehaviourBytes behavior)
   {
      String desc = behavior.getJvmSignature();
      String[] rtn = new String[1];
      getParameterTypeString(desc, desc.indexOf(')') + 1, rtn, 0);
      return rtn[0];
   }
   
   public static String getFieldType(FieldBytes field)
   {
      String[] rtn = new String[1];
      getParameterTypeString(field.getJvmSignature(), 0, rtn, 0);
      return rtn[0];
   }
   
   /**
    * Put the next argument into the parameter string array 
    * 
    * @param desc the beahviour's descriptor
    * @param i the current index in the descriptor
    * @param args the parameter string array
    * @param n the current index of the arguments array
    */
   private static int getParameterTypeString(String desc, int i, String[] args, int n)
   {
      int i2;
      String name;

      int arrayDim = 0;
      char c = desc.charAt(i);
      if (c == ')')
         return ++i;
      while (c == '[')
      {
         ++arrayDim;
         c = desc.charAt(++i);
      }

      boolean object = false;
      if (c == 'L')
      {
         i2 = desc.indexOf(';', ++i);
         name = desc.substring(i, i2++).replace('/', '.');
         object = true;
      }
      else
      {
         String type = arrayDim == 0 ? toPrimitiveName(c) : String.valueOf(c);

         i2 = i + 1;
         if (arrayDim == 0)
         {
            args[n] = type;
            return i2; // neither an array type or a class type
         }
         else
            name = type;
      }

      if (arrayDim > 0)
      {
         StringBuilder sb = new StringBuilder();
         while (arrayDim-- > 0)
            sb.append("[");

         if (object)
            sb.append("L");
         sb.append(name);
         if (object)
            sb.append(";");

         name = sb.toString();
      }

      args[n] = name;
      return i2;
   }

   /**
    * Create a primitive entry
    * 
    * @param the character for the primitive
    * @return the primitive real name
    */
   private static String toPrimitiveName(char c)
   {
      BytecodePrimitive primitive = BytecodePrimitive.valueOf(String.valueOf(c));
      if (primitive == null)
         throw new IllegalArgumentException("Unknown primitive type " + c);
      return primitive.getName();
   }

   public String[] getParams()
   {
      return params;
   }
   
   public boolean isDouble(int i)
   {
      return params[i].equals("double");
   }
   
   public boolean isLong(int i)
   {
      return params[i].equals("long");
   }
   
   public static boolean isPrimitive(String param)
   {
      return BytecodePrimitive.valueOf(param) != null;
   }
}