/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.bytes.asm;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.jboss.reflect.plugins.bytecode.bytes.ClassBytes;
import org.jboss.reflect.plugins.bytecode.bytes.ClassBytesFactory;
import org.jboss.reflect.plugins.bytecode.bytes.PrimitiveBytes;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class AsmClassBytesFactory implements ClassBytesFactory
{
   ReadWriteLock lock = new ReentrantReadWriteLock();
   Map<ClassLoader, Map<String, ClassBytes>> cache = new WeakHashMap<ClassLoader, Map<String, ClassBytes>>(1);
   
   public static final AsmClassBytesFactory INSTANCE = new AsmClassBytesFactory();
   
   static final Map<String, PrimitiveBytes> PRIMITIVES;
   
   static
   {
      PRIMITIVES = new HashMap<String, PrimitiveBytes>();
      addPrimitive(AsmPrimitiveClassBytes.BOOLEAN);
      addPrimitive(AsmPrimitiveClassBytes.CHAR);
      addPrimitive(AsmPrimitiveClassBytes.BYTE);
      addPrimitive(AsmPrimitiveClassBytes.SHORT);
      addPrimitive(AsmPrimitiveClassBytes.INT);
      addPrimitive(AsmPrimitiveClassBytes.LONG);
      addPrimitive(AsmPrimitiveClassBytes.FLOAT);
      addPrimitive(AsmPrimitiveClassBytes.DOUBLE);
      addPrimitive(AsmPrimitiveClassBytes.VOID);
   }
   
   private AsmClassBytesFactory()
   {
      
   }
   
   static void addPrimitive(PrimitiveBytes primitive)
   {
      PRIMITIVES.put(primitive.getJvmName(), primitive);
      PRIMITIVES.put(primitive.getArrayComponentName(), primitive);
   }
   
   public ClassBytes loadPrimitive(String name)
   {
      return PRIMITIVES.get(name);
   }
   
   public ClassBytes loadClassBytes(ClassLoader initiating, String name)
   {
      if (initiating == null)
         throw new IllegalArgumentException("Null initiating classloader");
      if (name == null)
         throw new IllegalArgumentException("Null classloader");
               
      ClassLoader cl = initiating;
      
      ClassBytes clazz = findInCache(initiating, name);
      if (clazz != null)
         return clazz;
      
      int array = name.indexOf("[]", name.length() - 2);
      if (array > -1)
      {
         ClassBytes component = null;
         String componentName = name.substring(0, name.length() - 2);
         if (!componentName.endsWith("]"))
            component = loadPrimitive(name);
         if (component == null)
            component = loadClassBytes(initiating, componentName);
         if (component == null)
            return null;
         clazz = new AsmArrayBytes(component);
      }
      else
      {
         clazz = loadPrimitive(name);
         if (clazz != null)
            return clazz;
         
         URL url = initiating.getResource(name + ".class");
         if (url == null)
            return null;
         
         clazz = new AsmClassBytes(cl, url, name);
      }
      
      if (clazz != null)
         putInCache(initiating, name, clazz);
      
      return clazz;
   }
   
   private ClassBytes findInCache(ClassLoader loader, String name)
   {
      lockRead();
      try
      {
         Map<String, ClassBytes> classes = cache.get(loader);
         if (classes == null)
            return null;
         return classes.get(name);
      }
      finally
      {
         unlockRead();
      }
   }
   
   private void putInCache(ClassLoader loader, String name, ClassBytes clazz)
   {
      lockWrite();
      try
      {
         Map<String, ClassBytes> classes = cache.get(loader);
         if (classes == null)
         {
            classes = new HashMap<String, ClassBytes>(1);
            cache.put(loader, classes);
         }
         classes.put(name, clazz);
      }
      finally
      {
         unlockWrite();
      }
   }
   
   private void lockRead()
   {
      lock.readLock().lock();
   }
   
   private void unlockRead()
   {
      lock.readLock().unlock();
   }
   
   private void lockWrite()
   {
      lock.writeLock().lock();
   }
   
   private void unlockWrite()
   {
      lock.writeLock().unlock();
   }

}
