/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.bytecode;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.jboss.reflect.plugins.javassist.JavassistConstructor;
import org.jboss.reflect.plugins.javassist.JavassistConstructorInfo;
import org.jboss.reflect.plugins.javassist.JavassistField;
import org.jboss.reflect.plugins.javassist.JavassistFieldInfo;
import org.jboss.reflect.plugins.javassist.JavassistMethod;
import org.jboss.reflect.plugins.javassist.JavassistMethodInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.Strings;

/**
 * Class to wrap the generated JavassistMethod, -Constructor and -Field 
 * implementations with another implementation that checks the parameters
 * if something went wrong when invoking
 * 
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class ErrorCheckingMemberFactory
{
   /**
    * Wraps the generated JavassistMethod in an implementation that checks the 
    * parameters
    * 
    * @param m The wrapped method
    * @param method the target method for information about the parameters
    * @return the error checking wrapper 
    */
   static JavassistMethod wrapInErrorChecker(JavassistMethod m, JavassistMethodInfo method)
   {
      if (m == null || method == null)
         throw new IllegalArgumentException("Null method");
      
      int numParameters = method.getSignatureKey().getParams().length;
      
      return new ErrorCheckingJavassistMethod(m, method, numParameters);
   }
   
   /**
    * Wraps the generated JavassistConstructor in an implementation that checks the 
    * parameters
    * 
    * @param c The wrapped constructor
    * @param constructor the target constructor for information about the parameters
    * @return the error checking wrapper 
    */
   static JavassistConstructor wrapInErrorChecker(JavassistConstructor c, JavassistConstructorInfo constructor)
   {
      if (c == null || constructor == null)
         throw new IllegalArgumentException("Null constructor");
      
      int numParameters = constructor.getSignatureKey().getParams().length;
      
      return new ErrorCheckingJavassistConstructor(c, constructor, numParameters);
   }
   
   /**
    * Wraps the generated JavassistField in an implementation that checks the 
    * parameters
    * 
    * @param f The wrapped field
    * @param field the target field for information about the parameters
    * @return the error checking wrapper 
    */
   static JavassistField wrapInErrorChecker(JavassistField f, JavassistFieldInfo field)
   {
      if (f == null)
         throw new IllegalArgumentException("Null field");
      
      return new ErrorCheckingJavassistField(f, field);
   }
   
   private static boolean checkNumberOfParameters(Object[] args, int numParameters)
   {
      if (args == null && numParameters > 0)
         return false;
      if (args != null && args.length != numParameters)
         return false;
      return true;
   }
   
   private static void handleWrongParameters(String context, String target, Class<?>[] expected, Object[] args)
   {
      List<String> actual = new ArrayList<String>();
      if (args != null)
      {
         for (Object argument : args)
         {
            if (argument == null)
               actual.add(null);
            else
               actual.add(argument.getClass().getName());
         }
      }
      throw new IllegalArgumentException("Wrong arguments. " + context + " for target " + target + " expected=" + expected + " actual=" + actual);
      
   }

   private static void handleWrongTarget(Object target, Class<?> expected, String name)
   {
      throw new IllegalArgumentException("Wrong target for " + name + " " + target.getClass().getName() + " is not a " + expected.getName());
   }
   
   private static void handleNullTarget(AccessibleObject ao)
   {
      throw new IllegalArgumentException("Null target calling non-static " + ao);      
   }
   
   private static class ErrorCheckingJavassistMethod extends ErrorCheckingMemberFactory implements JavassistMethod
   {
      private final JavassistMethod delegate;
      private final JavassistMethodInfo method;
      private final int numParameters;
      private volatile Method real;
      
      private ErrorCheckingJavassistMethod(JavassistMethod delegate, JavassistMethodInfo method, int numParameters)
      {
         this.delegate = delegate;
         this.method = method;
         this.numParameters = numParameters;
      }

      public Object invoke(Object target, Object[] args) throws Throwable
      {
         if (!checkNumberOfParameters(args, numParameters))
            throw new IllegalArgumentException("Wrong number of parameters for " + method.getDeclaringClass().getName() + "." + method.getName() + method.getDescriptor());
         
         try
         {
            return delegate.invoke(target, args);
         }
         catch(ClassCastException e)
         {
            Method real = getRealMethod();
            if (!method.isStatic())
            {
               if (!real.getDeclaringClass().isAssignableFrom(target.getClass()))
                  handleWrongTarget(target, real.getDeclaringClass(), real.getName());
            }
            
            Class<?>[] params = real.getParameterTypes();
            for (int i = 0 ; i < args.length ; i++)
            {
               if (!params[i].isAssignableFrom(args[i].getClass()))
                  handleWrongParameters(method.getName(), Strings.defaultToString(target), real.getParameterTypes(), args);
            }
            
            throw e;
         }
         catch(NullPointerException e)
         {
            Method real = getRealMethod();
            if (!method.isStatic() && target == null)
               handleNullTarget(real);
            
            Class<?>[] parameters = real.getParameterTypes();
            for (int i = 0 ; i < parameters.length ; i++)
            {
               if (parameters[i].isPrimitive() && args[i] == null)
                  handleWrongParameters(method.getName(), Strings.defaultToString(target), real.getParameterTypes(), args);
            }
            
            throw e;
         }
      }
      
      private Method getRealMethod() throws Throwable
      {
         if (real == null)
         {
            Class<?> clazz = method.getDeclaringClass().getType();
            TypeInfo[] paramTypes = method.getParameterTypes();
            Class<?>[] params = new Class<?>[paramTypes.length];
            for (int i = 0 ; i < paramTypes.length ; i++)
            {
               params[i] = paramTypes[i].getType();
            }
            real = SecurityActions.getDeclaredMethod(clazz, method.getName(), params);
         }
         return real;
      }
      
   }

   private static class ErrorCheckingJavassistConstructor extends ErrorCheckingMemberFactory implements JavassistConstructor
   {
      private final JavassistConstructor delegate;
      private final JavassistConstructorInfo constructor;
      private final int numParameters;
      private volatile Constructor<?> real;
      
      private ErrorCheckingJavassistConstructor(JavassistConstructor delegate, JavassistConstructorInfo constructor, int numParameters)
      {
         this.delegate = delegate;
         this.constructor = constructor;
         this.numParameters = numParameters;
      }

      public Object newInstance(Object[] args) throws Throwable
      {
         if (!checkNumberOfParameters(args, numParameters))
            throw new IllegalArgumentException("Wrong number of parameters for " + constructor.getDeclaringClass() + "." + constructor.getName() + constructor.getDescriptor());
         
         try
         {
            return delegate.newInstance(args);
         }
         catch(ClassCastException e)
         {
            Constructor<?> real = getRealConstructor();
            Class<?>[] params = real.getParameterTypes();
            for (int i = 0 ; i < args.length ; i++)
            {
               if (!params[i].isAssignableFrom(args[i].getClass()))
                  handleWrongParameters("new", Strings.defaultToString(constructor.getDeclaringClass().getName()), real.getParameterTypes(), args);
            }
               
            throw e;
         }
         catch(NullPointerException e)
         {
            TypeInfo[] parameters = constructor.getParameterTypes();
            for (int i = 0 ; i < parameters.length ; i++)
            {
               if (parameters[i].isPrimitive() && args[i] == null)
               {
                  Constructor<?> real = getRealConstructor();
                  handleWrongParameters("new", Strings.defaultToString(real.getDeclaringClass().getName()), real.getParameterTypes(), args);
               }
            }
            
            throw e;
         }
      }
      
      private Constructor<?> getRealConstructor() throws Throwable
      {
         if (real == null)
         {
            Class<?> clazz = constructor.getDeclaringClass().getType();
            TypeInfo[] paramTypes = constructor.getParameterTypes();
            Class<?>[] params = new Class<?>[paramTypes.length];
            for (int i = 0 ; i < paramTypes.length ; i++)
            {
               params[i] = paramTypes[i].getType();
            }
            real = SecurityActions.getDeclaredConstructor(clazz, params);
         }
         return real;
      }
   }

   private static class ErrorCheckingJavassistField extends ErrorCheckingMemberFactory implements JavassistField
   {
      private final JavassistField delegate;
      private final JavassistFieldInfo field;
      private volatile Field real;
      
      private ErrorCheckingJavassistField(JavassistField delegate, JavassistFieldInfo field)
      {
         this.delegate = delegate;
         this.field = field;
      }

      public Object get(Object target) throws Throwable
      {
         try
         {
            return delegate.get(target);
         }
         catch (ClassCastException e)
         {
            Field real = getRealField();
            
            if (!field.isStatic() && !real.getDeclaringClass().isAssignableFrom(target.getClass()))
               handleWrongTarget(target, real.getDeclaringClass(), real.getName());
            
            throw e;
         }
         catch(NullPointerException e)
         {
            if (!field.isStatic() && target == null)
            {
               Field real = getRealField();
               handleNullTarget(real);
            }
            throw e;
         }
      }

      public void set(Object target, Object value) throws Throwable
      {
         try
         {
            delegate.set(target, value);
         }
         catch (ClassCastException e)
         {
            Field real = getRealField();
            Class<?> type = real.getType();

            if (!field.isStatic() && !real.getDeclaringClass().isAssignableFrom(target.getClass()))
               handleWrongTarget(target, real.getDeclaringClass(), real.getName());

            if (!type.isAssignableFrom(value.getClass()))
               throw new IllegalArgumentException("Wrong arguments. Setting " + real.getName() + " for target " + target + " expected=" + real.getType() + " actual=" + value.getClass());
         }
         catch(NullPointerException e)
         {
            
            if (!field.isStatic() && target == null)
            {
               Field real = getRealField();
               handleNullTarget(real);
            }
            if (field.getType().isPrimitive() && value == null)
               throw new IllegalArgumentException("Null value setting non-static field. " + real);
            throw e;
         }
      }
 
      private Field getRealField() throws Throwable
      {
         if (real == null)
         {
            Class<?> clazz = field.getDeclaringClass().getType();
            real = SecurityActions.getDeclaredField(clazz, field.getName());
         }
         return real;
      }
   
   }
   
   
}