/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.bootstrap.microcontainer;

import java.net.URL;
import java.util.Collections;
import java.util.List;

import org.jboss.beans.metadata.plugins.AbstractBeanMetaData;
import org.jboss.beans.metadata.plugins.InstallCallbackMetaData;
import org.jboss.beans.metadata.plugins.UninstallCallbackMetaData;
import org.jboss.beans.metadata.spi.CallbackMetaData;
import org.jboss.bootstrap.AbstractServerImpl;
import org.jboss.bootstrap.BootstrapMetaData;
import org.jboss.bootstrap.xml.BootstrapParser;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.plugins.bootstrap.basic.BasicBootstrap;
import org.jboss.kernel.plugins.deployment.xml.BasicXMLDeployer;
import org.jboss.kernel.plugins.event.AbstractEvent;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.event.KernelEvent;
import org.jboss.kernel.spi.event.KernelEventManager;
import org.jboss.util.StopWatch;

/**
 * ServerImpl.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class ServerImpl extends AbstractServerImpl
{
   /** The bootstrap file */
   public static String BOOTSTRAP_XML_NAME = "bootstrap.xml";

   /** The bootstrap */
   private BasicBootstrap bootstrap;
   
   /** The kernel */
   private Kernel kernel;
   
   /** The kernel deployer */
   private BasicXMLDeployer kernelDeployer;

   /**
    * Get the kernel
    * 
    * @return the kernel
    */
   public Kernel getKernel()
   {
      return kernel;
   }
   
   @Override
   protected void doStart(StopWatch watch) throws Throwable
   {
      // Bootstrap the kernel
      BasicBootstrap bootstrap = new BasicBootstrap();
      bootstrap.run();
      kernel = bootstrap.getKernel();

      // Register the server implementation
      KernelController controller = kernel.getController();
      AbstractBeanMetaData metaData = new AbstractBeanMetaData("JBossServer", getClass().getName());
      InstallCallbackMetaData install = new InstallCallbackMetaData();
      install.setMethodName("addBootstrap");
      metaData.setInstallCallbacks(Collections.singletonList((CallbackMetaData) install));
      UninstallCallbackMetaData uninstall = new UninstallCallbackMetaData();
      uninstall.setMethodName("removeBootstrap");
      metaData.setUninstallCallbacks(Collections.singletonList((CallbackMetaData) uninstall));
      controller.install(metaData, this);

      // Determine the url for the bootstrap
      URL configURL = getConfig().getServerConfigURL();
      URL url = getConfig().getBootstrapURL();
      if (url == null)
         url = new URL(configURL, BOOTSTRAP_XML_NAME);
      log.info("Starting Microcontainer, bootstrapURL=" + url);

      // Parse the bootstrap metadata
      BootstrapMetaData bootstrapMetaData = BootstrapParser.parse(url);
      List<String> bootstrapURLs = bootstrapMetaData.getBootstrapURLs();
      if (bootstrapURLs == null || bootstrapURLs.isEmpty())
         throw new IllegalStateException("No bootstrap urls in " + url);
      log.debug("BootstrapURLs=" + bootstrapURLs);
      
      // Create an xml deployer
      kernelDeployer = new BasicXMLDeployer(kernel);
      try
      {
         // Deploy the bootstrap urls
         for (String bootstrapURL : bootstrapURLs)
         {
            url = new URL(configURL, bootstrapURL);
            kernelDeployer.deploy(url);
         }
         
         // Check it is complete
         kernelDeployer.validate();
         
         // Send a notification that the startup is complete
         KernelEventManager eventMgr = kernel.getEventManager();
         KernelEvent startEvent = new AbstractEvent(eventMgr, START_NOTIFICATION_TYPE, 0, System.currentTimeMillis(), new Long(watch.getLapTime()));
         eventMgr.fireKernelEvent(startEvent);

         this.bootstrap = bootstrap;
      }
      catch (Throwable t)
      {
         try
         {
            kernelDeployer.shutdown();
         }
         catch (Throwable ignored)
         {
         }
         throw t;
      }
   }

   @Override
   protected void doShutdown()
   {
      // Send a stop event msg
      try
      {
         if (bootstrap != null)
         {
            KernelEvent stopEvent = bootstrap.createEvent(STOP_NOTIFICATION_TYPE, null);
            bootstrap.fireKernelEvent(stopEvent);
         }
      }
      catch (Throwable t)
      {
         log.warn("Error sending stop notification", t);
      }
      
      // Shutdown the deployer
      try
      {
         if (kernelDeployer != null)
         {
            kernelDeployer.shutdown();
            kernelDeployer = null;
         }
      }
      catch (Throwable t)
      {
         log.warn("Error stopping xml deployer", t);
      }
      
      // Shutdown the controller
      try
      {
         Kernel currentKernel = null;
         if (kernel != null)
         {
            currentKernel = kernel;
            kernel = null;
         }
         else if (bootstrap != null)
         {
            currentKernel = bootstrap.getKernel();
         }
         bootstrap = null;
         
         if (currentKernel != null)
         {
            KernelController controller = currentKernel.getController();
            controller.shutdown();
         }
      }
      catch (Throwable t)
      {
         log.warn("Error stopping xml deployer", t);
      }
   }
}
