/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.test.cluster.defaultcfg.simpleweb.test;

import static org.jboss.test.cluster.testutil.SessionTestUtil.validateExpectedAttributes;
import static org.jboss.test.cluster.testutil.SessionTestUtil.validateNewSession;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpSessionBindingEvent;
import javax.servlet.http.HttpSessionBindingListener;

import junit.framework.TestCase;

import org.jboss.cache.Fqn;
import org.jboss.cache.pojo.PojoCache;
import org.jboss.logging.Logger;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.test.cluster.testutil.JGroupsSystemPropertySupport;
import org.jboss.test.cluster.testutil.SessionTestUtil;
import org.jboss.test.cluster.web.mocks.BasicRequestHandler;
import org.jboss.test.cluster.web.mocks.InvalidateSessionRequestHandler;
import org.jboss.test.cluster.web.mocks.SetAttributesRequestHandler;
import org.jboss.web.tomcat.service.session.JBossCacheManager;

/**
 * A FailedExpirationTestCase.
 * 
 * @author Brian Stansberry
 * @version $Revision: 1.1 $
 */
public class FailedExpirationTestCase extends TestCase
{
   private static final Logger log = Logger.getLogger(ConcurrentFailoverRequestsTestCase.class);
   
   private static long testCount = System.currentTimeMillis();
   
   private final JGroupsSystemPropertySupport jgSupport = new JGroupsSystemPropertySupport();
   private Set<PojoCache> caches = new HashSet<PojoCache>();
   private String tempDir;

   /**
    * Create a new FailedExpirationTestCase.
    * 
    * @param name
    */
   public FailedExpirationTestCase(String name)
   {
      super(name);
   }

   @Override
   protected void setUp() throws Exception
   {
      super.setUp();
      
      jgSupport.setUpProperties();
      
      File tmpDir = new File(System.getProperty("java.io.tmpdir"));
      File root = new File(tmpDir, getClass().getSimpleName());
      root.mkdirs();
      root.deleteOnExit();
      tempDir = root.getAbsolutePath();
   }

   @Override
   protected void tearDown() throws Exception
   {
      try
      {
         super.tearDown();
      }
      finally
      {      
         jgSupport.restoreProperties();
         
         SessionTestUtil.clearDistributedCacheManagerFactory();
         
         for (PojoCache cache : caches)
         { 
            // Try to clean up so we avoid loading sessions 
            // from storage in later tests
            try
            {
               log.info("Removing /JSESSION from " + cache.getCache().getLocalAddress());
               cache.getCache().removeNode(Fqn.fromString("/JSESSION"));
            }
            catch (Exception e)
            {
               log.error("Cache " + cache.getCache().getLocalAddress() + ": " + e.getMessage(), e);
            }
            
            try
            {
               cache.stop();
               cache.destroy();
            }
            catch (Exception e)
            {
               log.error("Cache " + cache.getCache().getLocalAddress() + ": " + e.getMessage(), e);
            }
            
         }
         
         caches.clear();
         
         SessionTestUtil.cleanFilesystem(tempDir);
      }
   }
   
   public void testFailedInvalidation() throws Exception
   {
      ++testCount;
      
      JBossWebMetaData webMetaData = SessionTestUtil.createWebMetaData(10, true, 1, -1);
      webMetaData.getReplicationConfig().setMaxUnreplicatedInterval(0);
      String warName = "test" + testCount;
      JBossCacheManager jbcm = SessionTestUtil.createManager(warName, 2, true, null, false, false, null, caches);
      jbcm.init(warName, webMetaData);      
      jbcm.start();
      
      assertEquals(0, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());
      
      Map<String, Object> attrs = new HashMap<String, Object>(); 
      attrs.put("failInvalid", new FailInvalidationAttribute());
      attrs.put("failDeserialization", new FailDeserializationAttribute());
      
      // Establish session.
      SetAttributesRequestHandler setHandler = new SetAttributesRequestHandler(attrs, false);
      SessionTestUtil.invokeRequest(jbcm, setHandler, null);
      validateNewSession(setHandler);
      String id = setHandler.getSessionId();
      assertEquals(1, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());
      
      InvalidateSessionRequestHandler invalHandler = new InvalidateSessionRequestHandler(attrs.keySet(), false);
      try
      {
         SessionTestUtil.invokeRequest(jbcm, invalHandler, id);
         fail("Invalidation not rejected");
      }
      catch (RejectedException ok) {}
      assertEquals(0, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());
   }
   
   public void testFailedExpiration() throws Exception
   {
      ++testCount;
      
      JBossWebMetaData webMetaData = SessionTestUtil.createWebMetaData(10, true, 1, -1);
      webMetaData.getReplicationConfig().setMaxUnreplicatedInterval(0);
      String warName = "test" + testCount;
      JBossCacheManager jbcm = SessionTestUtil.createManager(warName, 2, true, null, false, false, null, caches);
      jbcm.init(warName, webMetaData);      
      jbcm.start();
      
      assertEquals(0, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());
      
      Map<String, Object> attrs = new HashMap<String, Object>(); 
      attrs.put("failInvalid", new FailInvalidationAttribute());
      attrs.put("failDeserialization", new FailDeserializationAttribute());
      
      // Establish session.
      SetAttributesRequestHandler setHandler = new SetAttributesRequestHandler(attrs, false);
      SessionTestUtil.invokeRequest(jbcm, setHandler, null);
      validateNewSession(setHandler);
      String id = setHandler.getSessionId();
      assertEquals(1, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());
      
      SessionTestUtil.sleepThread(2010);
      jbcm.backgroundProcess();
      assertEquals(0, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());      
   }
   
   public void testFailedExpirationAfterPassivation() throws Exception
   {
      ++testCount;
      
      String passivationDir = SessionTestUtil.getPassivationDir(tempDir, testCount, 1);
      JBossWebMetaData webMetaData = SessionTestUtil.createWebMetaData(10, true, 1, -1);
      webMetaData.getReplicationConfig().setMaxUnreplicatedInterval(0);
      String warName = "test" + testCount;
      JBossCacheManager jbcm = SessionTestUtil.createManager(warName, 2, true, passivationDir, false, false, null, caches);
      jbcm.init(warName, webMetaData);      
      jbcm.start();
      
      assertEquals(0, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());
      
      Map<String, Object> attrs = new HashMap<String, Object>(); 
      attrs.put("failInvalid", new FailInvalidationAttribute());
      attrs.put("failDeserialization", new FailDeserializationAttribute());
      
      // Establish session.
      SetAttributesRequestHandler setHandler = new SetAttributesRequestHandler(attrs, false);
      SessionTestUtil.invokeRequest(jbcm, setHandler, null);
      validateNewSession(setHandler);
      String id = setHandler.getSessionId();
      assertEquals(1, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());
      
      SessionTestUtil.sleepThread(1010);
      jbcm.backgroundProcess();
      assertEquals(0, jbcm.getActiveSessionCount());
      assertEquals(1, jbcm.getPassivatedSessionCount());   
      
      SessionTestUtil.sleepThread(1010);
      jbcm.backgroundProcess();
      assertEquals(0, jbcm.getActiveSessionCount());
      assertEquals(0, jbcm.getPassivatedSessionCount());     
   }
   
   public static class FailInvalidationAttribute extends FailDeserializationAttribute
      implements HttpSessionBindingListener
   {
      /** The serialVersionUID */
      private static final long serialVersionUID = 1L;

      public void valueBound(HttpSessionBindingEvent event)
      {
         // no-op
      }

      public void valueUnbound(HttpSessionBindingEvent arg0)
      {
         throw new RejectedException();
      }
   }
   
   public static class FailDeserializationAttribute implements Serializable
   {
      /** The serialVersionUID */
      private static final long serialVersionUID = 1L;
      
      private void readObject(java.io.ObjectInputStream in)
            throws IOException, ClassNotFoundException
      {
         throw new RejectedException();
      }
      
   }
   
   public static class RejectedException extends RuntimeException
   {
      /** The serialVersionUID */
      private static final long serialVersionUID = 1L;
      
   }

}
