/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.weld.integration.deployer.jndi;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.enterprise.inject.spi.BeanManager;
import javax.naming.Context;
import javax.naming.NamingException;

import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.reloaded.naming.deployers.javaee.JavaEEModuleInformer;
import org.jboss.util.naming.NonSerializableFactory;
import org.jboss.weld.bootstrap.spi.BeanDeploymentArchive;
import org.jboss.weld.integration.deployer.env.helpers.BootstrapBean;
import org.jboss.weld.manager.api.WeldManager;

/**
 * This deployer intercepts BootstrapBean metadata,
 * and adds JndiBinder invocations to it.
 *
 * @author Pete Muir
 * @author <a href="mailto:stan.silvert@jboss.org">Stan Silvert</a>
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class JndiBinderDeployer extends AbstractJndiBinderDeployer<BootstrapBean>
{
   public JndiBinderDeployer()
   {
      super(BootstrapBean.class);
      this.setWantComponents(false);
   }

   private BeanDeploymentArchiveLocator beanDeploymentArchiveLocator;

   private JavaEEModuleInformer moduleInformer;

   public void setBeanDeploymentArchiveLocator(BeanDeploymentArchiveLocator beanDeploymentArchiveLocator)
   {
      this.beanDeploymentArchiveLocator = beanDeploymentArchiveLocator;
   }

   public void setModuleInformer(JavaEEModuleInformer moduleInformer)
   {
      this.moduleInformer = moduleInformer;
   }

   // --- binding logic ---

   public void bind(BootstrapBean bootstrapBean, DeploymentUnit deploymentUnit) throws NamingException
   {
      Context context = getBeanManagerContext();

      if (moduleInformer.getModuleType(deploymentUnit).equals(JavaEEModuleInformer.ModuleType.EJB) ||
            moduleInformer.getModuleType(deploymentUnit).equals(JavaEEModuleInformer.ModuleType.WEB))
      {
         BeanDeploymentArchive beanDeploymentArchive = beanDeploymentArchiveLocator.getBeanDeploymentArchive(bootstrapBean, deploymentUnit);
         if (beanDeploymentArchive != null)
         {
            String applicationName = moduleInformer.getApplicationName(deploymentUnit);
            String name = (applicationName == null) ? "" : applicationName + "/";
            name += moduleInformer.getModulePath(deploymentUnit);
            NonSerializableFactory.rebind(context, name, bootstrapBean.getBootstrap().getManager(beanDeploymentArchive), true);
         }
      }
   }

   public void unbind(BootstrapBean bootstrapBean, DeploymentUnit deploymentUnit) throws NamingException
   {
      Map<String, BeanManager> beanManagers = new DeploymentVisitor(bootstrapBean).visit().getBeanManagers();
      Context context = (Context)getBeanManagerContext();
      String applicationName = moduleInformer.getApplicationName(deploymentUnit);
      String topContext = applicationName != null? applicationName: moduleInformer.getModulePath(deploymentUnit);
      try
      {
         Context appContext = (Context) context.lookup(topContext);
         context.destroySubcontext(topContext);
      }
      catch (NamingException e)
      {
         // ignore if missing
      }

   }

   protected static class DeploymentVisitor
   {
      private final Map<String, BeanManager> beanManagers;
      private final BootstrapBean bootstrapBean;

      public DeploymentVisitor(BootstrapBean bootstrapBean)
      {
         this.beanManagers = new HashMap<String, BeanManager>();
         this.bootstrapBean = bootstrapBean;
      }

      public Map<String, BeanManager> getBeanManagers()
      {
         return Collections.unmodifiableMap(beanManagers);
      }

      public DeploymentVisitor visit()
      {
         for (BeanDeploymentArchive bda : bootstrapBean.getDeployment().getBeanDeploymentArchives())
         {
            visit(bda, new HashSet<BeanDeploymentArchive>());
         }
         return this;
      }

      private void visit(BeanDeploymentArchive bda, Set<BeanDeploymentArchive> seenBdas)
      {
         WeldManager beanManager = bootstrapBean.getBootstrap().getManager(bda);
         // BeanManager id is simply the BDA ID
         String key = beanManager.getId();
         beanManagers.put(key, beanManager);
         seenBdas.add(bda);
         for (BeanDeploymentArchive child : bda.getBeanDeploymentArchives())
         {
            if (!seenBdas.contains(child))
            {
               visit(child, seenBdas);
            }
         }
      }

   }
}