/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.lang;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Usefull methods for classes.
 *
 * @author <a href="mailto:carlo.dewolf@jboss.com">Carlo de Wolf</a>
 * @version $Revision: 68152 $
 */
public class ClassHelper
{
   /**
    * @see Class#argumentTypesToString
    */
   private static String argumentTypesToString(Class<?>[] argTypes)
   {
      StringBuilder buf = new StringBuilder();
      buf.append("(");
      if (argTypes != null)
      {
         for (int i = 0; i < argTypes.length; i++)
         {
            if (i > 0)
            {
               buf.append(", ");
            }
            Class<?> c = argTypes[i];
            buf.append((c == null) ? "null" : c.getName());
         }
      }
      buf.append(")");
      return buf.toString();
   }

   private static Method findPrivateMethod(Class<?> target, String methodName, Class<?>... paramTypes)
   {
      // Top of the world
      if(target == null)
         return null;
      
      // TODO: what is faster? scan or catch exception?
      for(Method method : target.getDeclaredMethods())
      {
         if(method.getName().equals(methodName))
         {
            if(paramTypes == null)
               return method;
            
            if(Arrays.equals(method.getParameterTypes(), paramTypes))
               return method;
         }
      }
      
      return findPrivateMethod(target.getSuperclass(), methodName, paramTypes);
   }
   
   /**
    * Find all methods with a specific name on a class and it's super classes
    * regardless of parameter signature.
    * 
    * @param cls            the class to scan
    * @param methodName     the name of the methods to find
    * @return               a list of methods found, or empty
    */
   public static List<Method> getAllMethodsByName(Class<?> cls, String methodName)
   {
      List<Method> methods = new ArrayList<Method>();
      populateWithMethodsByName(methods, cls, methodName);
      return methods;
   }
   
   /**
    * Find all methods starting with the specified prefix on the specified
    * class
    * 
    * @param clazz
    * @param methodNamePrefix
    * @return
    */
   public static List<Method> getAllMethodsByPrefix(Class<?> clazz, String methodNamePrefix)
   {
      List<Method> methods = new ArrayList<Method>();
      ClassHelper.populateWithMethodsByPrefix(methods, clazz, methodNamePrefix);
      return methods;
   }
   
   /**
    * Returns the <code>Method</code> with the given attributes of either this class
    * or one of it's super classes.
    * 
    * TODO: return type specifics are not considered
    * FIXME: rename method (it must return all modifiers)
    * 
    * @param cls            class to scan
    * @param methodName     the name of the method
    * @param paramTypes     the parameter types
    * @return               the <code>Method</code> matching the method name and parameters
    * @throws NoSuchMethodException if no method can be found
    */
   public static Method getPrivateMethod(Class<?> cls, String methodName, Class<?>... paramTypes) throws NoSuchMethodException
   {
      assert cls != null : "cls is null";
      assert methodName != null : "methodName is null";
      
      Method result = findPrivateMethod(cls, methodName, paramTypes);
      if(result == null)
         throw new NoSuchMethodException(cls.getName() + "." + methodName + argumentTypesToString(paramTypes));
      
      return result;
   }
   
   private static void populateWithMethodsByName(List<Method> methods, Class<?> cls, String methodName)
   {
      // Top of the world
      if(cls == null)
         return;
      
      for(Method method : cls.getDeclaredMethods())
      {
         if(method.getName().equals(methodName))
            methods.add(method);
      }
      
      populateWithMethodsByName(methods, cls.getSuperclass(), methodName);
   }
   
   private static void populateWithMethodsByPrefix(List<Method> methods, Class<?> clazz, String methodNamePrefix)
   {
      // Exit Condition
      if (clazz == null)
      {
         return;
      }

      // For all declared methods
      for (Method method : clazz.getDeclaredMethods())
      {
         if (method.getName().startsWith(methodNamePrefix))
            methods.add(method);
      }

      populateWithMethodsByPrefix(methods, clazz.getSuperclass(), methodNamePrefix);
   }
}
