/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.management.upload.remoting;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import javax.management.MBeanServer;

import org.jboss.aop.joinpoint.Invocation;
import org.jboss.aop.joinpoint.InvocationResponse;
import org.jboss.aspects.remoting.AOPRemotingInvocationHandler;
import org.jboss.deployers.client.spi.main.MainDeployer;
import org.jboss.deployers.spi.management.deploy.DeploymentID;
import org.jboss.deployers.spi.management.deploy.DeploymentManager;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.management.upload.SerializableDeploymentID;
import org.jboss.profileservice.spi.DeploymentRepository;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.stream.StreamInvocationHandler;
import org.jboss.virtual.VirtualFile;

/**
 * A remoting StreamInvocationHandler installed as the profile service subsystem
 * handler and used by the StreamingDeploymentTarget implementation. 
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision:$
 */
public class DeployHandler extends AOPRemotingInvocationHandler
   implements StreamInvocationHandler
{
   static Logger log = Logger.getLogger(DeployHandler.class);
   private DeploymentRepository deploymentRepository;
   /** The deployment factory */
   private VFSDeploymentFactory deploymentFactory = VFSDeploymentFactory.getInstance();
   private MainDeployer mainDeployer;

   
   public DeploymentRepository getDeploymentRepository()
   {
      return deploymentRepository;
   }
   public void setDeploymentRepository(DeploymentRepository deploymentRepository)
   {
      this.deploymentRepository = deploymentRepository;
   }

   
   public VFSDeploymentFactory getDeploymentFactory()
   {
      return deploymentFactory;
   }
   public void setDeploymentFactory(VFSDeploymentFactory deploymentFactory)
   {
      this.deploymentFactory = deploymentFactory;
   }

   public MainDeployer getMainDeployer()
   {
      return mainDeployer;
   }
   public void setMainDeployer(MainDeployer mainDeployer)
   {
      this.mainDeployer = mainDeployer;
   }
   /**
    * Handle a DeploymentManager distribute invocation
    * @see DeploymentManager#distribute(String, DeploymentPhase, java.net.URL)
    * @param request - the remoting invocation
    */
   public Object handleStream(InputStream contentIS, InvocationRequest request)
      throws Throwable
   {
      SerializableDeploymentID deploymentTarget = (SerializableDeploymentID) request.getParameter();
      log.debug("handleStream, deploymentTarget: "+deploymentTarget);
      deploymentTarget.setContentIS(contentIS);
      DeploymentPhase phase = deploymentTarget.getPhase();
      String[] names = deploymentTarget.getNames();
      String repositoryName = deploymentRepository.addDeploymentContent(names[0], contentIS, phase);
      String[] rnames = {repositoryName};
      deploymentTarget.setRepositoryNames(rnames);
      return new InvocationResponse(repositoryName);
   }

   public void addListener(InvokerCallbackHandler arg0)
   {      
   }

   /**
    * Handle a DeploymentManager invocation other than distribute
    * @param request - the remoting invocation
    * @return the result of the invocation
    */
   public Object invoke(InvocationRequest request) throws Throwable
   {
      Object parameter = request.getParameter();
      Object returnValue = null;

      if(parameter instanceof Invocation)
      {
         returnValue = super.invoke(request);
      }
      else
      {
         Map payload = request.getRequestPayload();
         DeploymentID dtID = (DeploymentID) payload.get("DeploymentTargetID");
         log.debug("invoke, payload: "+payload);
         if( parameter.equals("getRepositoryNames"))
         {
            String[] names = (String[]) payload.get("names");
            DeploymentPhase phase = (DeploymentPhase) payload.get("phase");
            returnValue = getRepositoryNames(names, phase);
         }
         if( parameter.equals("start") )
         {
            start(dtID);
         }
         if( parameter.equals("stop") )
         {
            stop(dtID);
         }
         if( parameter.equals("undeploy") )
         {
            undeploy(dtID);
         }
      }
      return returnValue;
   }

   public void removeListener(InvokerCallbackHandler arg0)
   {
   }

   public void setInvoker(ServerInvoker arg0)
   {
   }

   public void setMBeanServer(MBeanServer arg0)
   {
   }

   protected String[] getRepositoryNames(String[] names, DeploymentPhase phase)
      throws IOException
   {
      return deploymentRepository.getRepositoryNames(names, phase);
   }

   protected void start(DeploymentID dtID)
      throws Exception
   {
      String[] names = dtID.getNames();
      log.info("Begin start, "+Arrays.asList(names));
      // Prevent hot deployment scans from seeing in transition deployments
      deploymentRepository.acquireDeploymentContentLock();
      try
      {
         for(String name : names)
         {
            VirtualFile vf = deploymentRepository.getDeploymentContent(name, dtID.getPhase());
            VFSDeployment vfsd = createDeployment(vf);
            deploymentRepository.addDeployment(name, vfsd, dtID.getPhase());
            deploymentRepository.unlockDeploymentContent(vf.getPathName(), dtID.getPhase());
            mainDeployer.addDeployment(vfsd);
            log.info("Scheduling start for: "+vfsd);
         }
         mainDeployer.process();
         mainDeployer.checkComplete();
      }
      finally
      {
         deploymentRepository.releaseDeploymentContentLock();
      }
      log.info("End start, "+Arrays.asList(names));
   }

   protected void stop(DeploymentID dtID)
      throws Exception
   {
      String[] names = dtID.getNames();
      log.info("stop, "+Arrays.asList(names));
      for(String name : names)
      {
         VFSDeployment vfsd = deploymentRepository.getDeployment(name, dtID.getPhase());
         mainDeployer.removeDeployment(vfsd);
         log.info("Scheduling stop for: "+vfsd);
      }
      mainDeployer.process();
      mainDeployer.checkComplete();
   }

   protected void undeploy(DeploymentID dtID)
      throws Exception
   {
      String[] names = dtID.getNames();
      log.info("undeploy, "+Arrays.asList(names));
      for(String name : names)
      {
         deploymentRepository.removeDeployment(name, dtID.getPhase());
         log.info("Undeployed: "+name);
      }
   }

   /**
    * Create a deployment
    * 
    * @param file the root file
    * @return the deployment
    */
   protected VFSDeployment createDeployment(VirtualFile file)
   {
      if (file == null)
         throw new IllegalArgumentException("Null file");
      return deploymentFactory.createVFSDeployment(file);
   }
}
