/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.mock.ds;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import javax.management.ObjectName;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.managed.ManagedObjectCreator;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.deployer.JAXPDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.plugins.DefaultFieldsImpl;
import org.jboss.managed.plugins.ManagedObjectImpl;
import org.jboss.managed.plugins.ManagedPropertyImpl;
import org.jboss.metatype.api.types.SimpleMetaType;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValueSupport;
import org.jboss.profileservice.management.builders.ServiceManagedObject;
import org.jboss.system.metadata.ServiceAttributeMetaData;
import org.jboss.system.metadata.ServiceConstructorMetaData;
import org.jboss.system.metadata.ServiceDependencyMetaData;
import org.jboss.system.metadata.ServiceDeployment;
import org.jboss.system.metadata.ServiceMetaData;
import org.jboss.system.metadata.ServiceTextValueMetaData;
import org.jboss.virtual.VirtualFile;
import org.w3c.dom.Document;

/**
 * A fake datasource deployer (*-dsf.xml, subset of *-ds.xml) for testing
 * the profile service spis. This transforms the *-dsf.xml into a single
 * ServiceMetaData representing the DataSource connection mbean.
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision:$
 */
public class FakeDataSourceDeployer extends JAXPDeployer<ServiceDeployment>
   implements ManagedObjectCreator
{  
   /**
    * A mapping from the managed property name to the attachment property name.
    */
   private Map<String, String> propertyNameMappings = new HashMap<String, String>();

   public FakeDataSourceDeployer()
   {
      super(ServiceDeployment.class);
      setSuffix("-dsf.xml");
   }

   public Map<String, String> getPropertyNameMappings()
   {
      return propertyNameMappings;
   }

   public void setPropertyNameMappings(Map<String, String> mangedToMBeanNames)
   {
      // Need to reverse the mapping
      for(String key : mangedToMBeanNames.keySet())
      {
         String value = mangedToMBeanNames.get(key);
         propertyNameMappings.put(value, key);
      }
   }

   public void build(DeploymentUnit unit, Map<String, ManagedObject> map) 
      throws DeploymentException
   {
      String name = unit.getSimpleName();
      if(name.endsWith("-dsf.xml"))
      {
         Map<String, Object> attachments = unit.getTransientManagedObjects().getAttachments();
         log.info(name+" attachments: "+attachments);
         ServiceDeployment service = unit.getAttachment(ServiceDeployment.class);
         if( service == null )
            throw new DeploymentException("Failed to find ServiceDeployment in "+unit.getName());
         List<ServiceMetaData> services = service.getServices();
         // TODO, can have multiple datasources in a deployment
         if( services.size() != 1 )
            throw new DeploymentException("Expected only 1 ServiceMetaData but saw "+services.size()+" in "+unit.getName());
         ServiceMetaData dsMetaData = services.get(0);
         String attachName = ServiceMetaData.class.getName();
         ManagedObject mo = new ServiceManagedObject(attachName, dsMetaData, propertyNameMappings);
         //ManagedObject wrapMO = WrapperAdvice.wrapManagedObject(mo);
         map.put(attachName, mo);
         // The datasource type...
         attachName = "FakeConnectionFactoryDeployer.datasource-type";
         ManagedObjectImpl typeMO = new ManagedObjectImpl(attachName);
         DefaultFieldsImpl f21 = new DefaultFieldsImpl();
         f21.setName("datasource-type");
         f21.setDescription("The type of the DataSource");
         f21.setMandatory(true);
         f21.setValue("local-tx-datasource");
         HashSet<MetaValue> values = new HashSet<MetaValue>();
         values.add(SimpleValueSupport.wrap("local-tx-datasource"));
         values.add(SimpleValueSupport.wrap("no-tx-datasource"));
         values.add(SimpleValueSupport.wrap("xa-datasource"));
         f21.setLegalValues(values);
         f21.setMetaType(SimpleMetaType.STRING);
         ManagedPropertyImpl dsType = new ManagedPropertyImpl(typeMO, f21);
         typeMO.getProperties().put("datasource-type", dsType);
         //wrapMO = WrapperAdvice.wrapManagedObject(typeMO);
         map.put(attachName, typeMO);
      }
   }

   @Override
   protected ServiceDeployment parse(VFSDeploymentUnit unit, VirtualFile file, Document document) throws Exception
   {
      ServiceDeployment deployment = new ServiceDeployment();
      DataSourceDeployment ds = new DataSourceDeployment();
      ds.parse(document);

      ServiceMetaData dsMbean = createDsServiceMetaData(ds);
      ArrayList<ServiceMetaData> services = new ArrayList<ServiceMetaData>();
      services.add(dsMbean);
      deployment.setServices(services);

      ComponentType type = new ComponentType("FakeDataSource", "LocalTx");
      unit.addAttachment(ComponentType.class, type);

      return deployment;
   }

   @Override
   protected boolean allowsReparse()
   {
      return true;
   }

   public ServiceMetaData createDsServiceMetaData(DataSourceDeployment ds)
      throws Exception
   {
      ServiceMetaData dsMbean = new ServiceMetaData();
      log.info("DataSource settings: "+ds);

      String jndiName = ds.getJndiName() == null ? "DefaultFakeDS" : ds.getJndiName();
      ObjectName objectName = new ObjectName("jboss.jca:type=FakeDataSourceConn,jndiName="+jndiName);
      dsMbean.setObjectName(objectName);
      dsMbean.setCode(FakeDataSourceConn.class.getName());
      // FakeDataSourceConn(DataSourceDeployment)
      ServiceConstructorMetaData constructor = new ServiceConstructorMetaData();
      constructor.setSignature(new String[] {DataSourceDeployment.class.getName()});
      constructor.setParameters(new Object[] {ds});
      dsMbean.setConstructor(constructor);

      // Attributes
      List<ServiceAttributeMetaData> attributes = new ArrayList<ServiceAttributeMetaData>();
      ServiceAttributeMetaData attribute = new ServiceAttributeMetaData();
      // jndiName
      attribute.setName("JndiName");
      attribute.setReplace(true);
      attribute.setTrim(true);
      attribute.setValue(new ServiceTextValueMetaData(jndiName));
      attributes.add(attribute);
      // jdbcURL
      attribute = new ServiceAttributeMetaData();
      attribute.setName("JdbcURL");
      if( ds.getJdbcURL() != null )
         attribute.setValue(new ServiceTextValueMetaData(ds.getJdbcURL()));
      attributes.add(attribute);
      // driverClass
      attribute = new ServiceAttributeMetaData();
      attribute.setName("DriverClass");
      if(ds.getDriverClass() != null)
         attribute.setValue(new ServiceTextValueMetaData(ds.getDriverClass()));
      attributes.add(attribute);
      // username
      attribute = new ServiceAttributeMetaData();
      attribute.setName("Username");
      if(ds.getUsername() != null)
         attribute.setValue(new ServiceTextValueMetaData(ds.getUsername()));
      attributes.add(attribute);
      // password
      attribute = new ServiceAttributeMetaData();
      attribute.setName("Password");
      if(ds.getPassword() != null)
         attribute.setValue(new ServiceTextValueMetaData(ds.getPassword()));
      attributes.add(attribute);
      // securityDomain
      attribute = new ServiceAttributeMetaData();
      attribute.setName("SecurityDomain");
      if(ds.getSecurityDomain() != null)
         attribute.setValue(new ServiceTextValueMetaData(ds.getSecurityDomain()));
      attributes.add(attribute);
      // minPoolSize
      attribute = new ServiceAttributeMetaData();
      attribute.setName("MinPoolSize");
      if(ds.getMinPoolSize() != null)
         attribute.setValue(new ServiceTextValueMetaData(ds.getMinPoolSize()));
      attributes.add(attribute);
      // maxPoolSize
      attribute = new ServiceAttributeMetaData();
      attribute.setName("MaxPoolSize");
      if(ds.getMaxPoolSize() != null)
         attribute.setValue(new ServiceTextValueMetaData(ds.getMaxPoolSize()));
      attributes.add(attribute);
      // connectionProps
      attribute = new ServiceAttributeMetaData();
      attribute.setName("ConnectionProperties");
      if(ds.getMaxPoolSize() != null)
         attribute.setValue(new ServiceTextValueMetaData(ds.getMaxPoolSize()));
      attributes.add(attribute);

      dsMbean.setAttributes(attributes);

      // Dependencies...Still have old jmx names here
      Collection<String> depends = ds.getDepends();
      List<ServiceDependencyMetaData> dependencies = new ArrayList<ServiceDependencyMetaData>();
      for(String iDependOn : depends)
      {
         ServiceDependencyMetaData sdmd = new ServiceDependencyMetaData();
         sdmd.setIDependOn(iDependOn);
      }
      dsMbean.setDependencies(dependencies);
      return dsMbean;
   }
}
